package v2

import (
	"encoding/json"
	"time"

	"code.justin.tv/web/cohesion/rpc"
)

// ListAssocResponseWithMeta is a representation of all associations between 2 entities
type ListAssocResponseWithMeta struct {
	From         Entity                   `json:"from"`
	Associations []*AssocResponseWithMeta `json:"associations"`
}

// AssocResponseWithMeta is a representation of an association between 2 entities
type AssocResponseWithMeta struct {
	Kind  string         `json:"kind"`
	Assoc *AssocResponse `json:"assoc"`
}

// AssocResponse is a representation of Cohesion's association
type AssocResponse struct {
	Entity  Entity                 `json:"target"`
	DataBag map[string]interface{} `json:"data"`
	Time    time.Time              `json:"time"`
	Cursor  string                 `json:"cursor"`
}

// ListResponse contains a list of the associations in response to a request
type ListResponse struct {
	Entity       Entity           `json:"from"`
	Kind         string           `json:"kind"`
	Associations []*AssocResponse `json:"to"`
	Cursor       string           `json:"cursor"`
}

// CountResponse contains the count in response to a request
type CountResponse struct {
	Count int
}

// newListResponse converts an rpc.ListResponseV2 into a ListResponse
// Returns nothing if unmarshalling a data bag errors out
func newListResponse(resp *rpc.ListResponseV2) *ListResponse {
	toCount := len(resp.To)
	cells := make([]*AssocResponse, toCount)
	for i, ar := range resp.To {
		var data map[string]interface{}
		if len(ar.Data) > 0 {
			err := json.Unmarshal(ar.Data, &data)
			if err != nil {
				return nil
			}
		}

		cells[i] = &AssocResponse{
			Entity:  Entity{ID: ar.Target.Name, Kind: ar.Target.Kind},
			DataBag: data,
			Time:    time.Unix(ar.Time.Seconds, ar.Time.Nanos),
			Cursor:  ar.Cursor,
		}
	}

	return &ListResponse{
		Entity:       Entity{ID: resp.From.Name, Kind: resp.From.Kind},
		Kind:         resp.Kind,
		Associations: cells,
		Cursor:       resp.Cursor,
	}
}

func newListAssocResponseWithMeta(resp *rpc.GetAllResponseV2) *ListAssocResponseWithMeta {
	var data map[string]interface{}
	ret := ListAssocResponseWithMeta{}
	ret.From = Entity{ID: resp.From.Name, Kind: resp.From.Kind}
	for _, assoc := range resp.To {
		assocResponseWithMeta := &AssocResponseWithMeta{}
		assocResponseWithMeta.Kind = assoc.Kind

		if len(assoc.Target.Data) > 0 {
			err := json.Unmarshal(assoc.Target.Data, &data)
			if err != nil {
				return nil
			}
		}
		assocResponseWithMeta.Assoc = &AssocResponse{
			Entity:  Entity{ID: assoc.Target.Target.Name, Kind: assoc.Target.Target.Kind},
			DataBag: data,
			Time:    time.Unix(assoc.Target.Time.Seconds, assoc.Target.Time.Nanos),
			Cursor:  assoc.Target.Cursor,
		}
		ret.Associations = append(ret.Associations, assocResponseWithMeta)
	}
	return &ret
}
