package populatefeed

import (
	"strconv"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"golang.org/x/net/context"
)

// StorageConfig configures RankedStorage
type StorageConfig struct {
	TableName *distconf.Str
}

// Load storage config from distconf
func (s *StorageConfig) Load(dconf *distconf.Distconf) error {
	s.TableName = dconf.Str("masonry.story_feed_table", "")
	if s.TableName.Get() == "" {
		return errors.New("unable to find storage table masonry.story_feed_table")
	}
	return nil
}

// A Story is everything stored in a row in the database
type Story struct {
	FeedID    string    `json:"feed_id"`
	StoryID   string    `json:"story_id"`
	Activity  Activity  `json:"activity"`
	Score     float64   `json:"score"`
	UpdatedAt time.Time `json:"updated_at"`
}

// Activity is the feed object that comes into fanout service
type Activity struct {
	Entity entity.Entity `json:"entity"`
	Verb   verb.Verb     `json:"verb"`
	Actor  entity.Entity `json:"actor"`
}

// StoryFeedStorage stores scored stories into DynamoDB
type StoryFeedStorage struct {
	Dynamo *dynamodb.DynamoDB
	Config *StorageConfig
	Log    *log.ElevatedLog
}

// SaveStory puts a scored item into dynamodb
func (s *StoryFeedStorage) SaveStory(ctx context.Context, feedID, storyID string, activity Activity, score float64) error {
	updatedAt := nowMicros()

	story := &Story{
		FeedID:    feedID,
		StoryID:   storyID,
		Score:     score,
		Activity:  activity,
		UpdatedAt: *updatedAt,
	}

	params := &dynamodb.PutItemInput{
		TableName: aws.String(s.Config.TableName.Get()),
		Item: map[string]*dynamodb.AttributeValue{
			"feed_id":    {S: &story.FeedID},
			"story_id":   {S: &story.StoryID},
			"actor":      {S: aws.String(story.Activity.Actor.Encode())},
			"verb":       {S: aws.String(string(story.Activity.Verb))},
			"entity":     {S: aws.String(story.Activity.Entity.Encode())},
			"score":      {N: aws.String(formatFloat(story.Score))},
			"updated_at": timeToAttributeValue(story.UpdatedAt),
		},
	}

	req, _ := s.Dynamo.PutItemRequest(params)
	return service_common.ContextSend(ctx, req, s.Log)
}

func timeToAttributeValue(t time.Time) *dynamodb.AttributeValue {
	nano := strconv.FormatInt(t.UTC().UnixNano(), 10)
	return &dynamodb.AttributeValue{N: aws.String(nano)}
}

func nowMicros() *time.Time {
	t := time.Now().UTC().Truncate(time.Microsecond)
	return &t
}

func formatFloat(f float64) string {
	return strconv.FormatFloat(f, 'f', -1, 64)
}
