package populatefeed

import (
	"fmt"
	"os"

	"code.justin.tv/feeds/consulconf"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/log/fmtlogger"
	"code.justin.tv/feeds/service-common"
	"github.com/hashicorp/consul/api"
)

type UtilityCommon struct {
	DistConfig  *distconf.Distconf
	Environment string
	Log         *log.ElevatedLog
}

type UtilityConfig struct {
	Team        string
	UtilityName string
}

func (s *UtilityCommon) Setup(utilityConfig *UtilityConfig) error {
	if err := s.setupEnvironment(); err != nil {
		return err
	}

	if err := s.setupDistConfig(utilityConfig); err != nil {
		return err
	}

	if err := s.setupLogging(); err != nil {
		return err
	}

	return nil
}

func (s *UtilityCommon) setupEnvironment() error {
	if s.Environment != "" {
		return nil
	}
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		return errors.New("unable to find ENVIRONMENT variable in env")
	}
	s.Environment = env
	return nil
}

func (s *UtilityCommon) setupDistConfig(utilityConfig *UtilityConfig) error {
	if s.DistConfig != nil {
		return nil
	}

	rootConfigs, err := s.rootConfigs(utilityConfig)
	if err != nil {
		return err
	}

	tmpDconf := distconf.Distconf{
		Readers: rootConfigs,
	}
	defer tmpDconf.Close()

	consulAddress := tmpDconf.Str("consul.address", "")
	consulDatacenter := tmpDconf.Str("consul.datacenter", "us-west2")
	if consulAddress.Get() == "" {
		return errors.New("unable to find consul.address location")
	}

	consulClientConfig := api.DefaultConfig()
	consulClientConfig.Address = consulAddress.Get()
	consulClientConfig.Datacenter = consulDatacenter.Get()

	consulClient, err := api.NewClient(consulClientConfig)
	if err != nil {
		return err
	}
	if _, err = consulClient.Agent().NodeName(); err != nil {
		return errors.Wrap(err, "unable to verify consul connection")
	}

	cconf1 := consulconf.Consulconf{
		KV:     consulClient.KV(),
		Prefix: fmt.Sprintf("settings/%s/%s/%s", utilityConfig.Team, utilityConfig.UtilityName, s.Environment),
	}
	cconf2 := consulconf.Consulconf{
		KV:     consulClient.KV(),
		Prefix: fmt.Sprintf("settings/%s/%s", utilityConfig.Team, s.Environment),
	}

	s.DistConfig = &distconf.Distconf{
		Readers: append(rootConfigs, &cconf1, &cconf2),
	}

	return nil
}

func (s *UtilityCommon) setupLogging() error {

	stdoutLog := fmtlogger.NewLogfmtLogger(os.Stdout, log.Discard)

	normalLogLevel := &service_common.StackLogger{
		RootLogger: stdoutLog,
	}

	s.Log = &log.ElevatedLog{
		NormalLog: log.ContextLogger{
			Dims:   &log.CtxDimensions{},
			Logger: normalLogLevel,
		},
		DebugLog: log.ContextLogger{
			Dims:   &log.CtxDimensions{},
			Logger: normalLogLevel,
		},
		LogToDebug: func(vals ...interface{}) bool {
			// TODO: Move context.Cancel to debug log
			return false
		},
	}
	s.Log.Debug("Debug log setup complete")
	return nil
}

func (s *UtilityCommon) rootConfigs(utilityConfig *UtilityConfig) ([]distconf.Reader, error) {

	memConfig := distconf.InMemory{}
	memConfig.Write("consul.address", []byte("consul.internal.justin.tv"))
	memConfig.Write("consul.datacenter", []byte("us-west2"))

	cmdLineConfig := &distconf.CommandLine{Prefix: "--conf:"}

	envConfig := distconf.Env{
		Prefix:   utilityConfig.UtilityName + "_",
		OsGetenv: os.Getenv,
	}

	return []distconf.Reader{cmdLineConfig, &envConfig, &memConfig}, nil
}
