package main

import (
	"context"
	"flag"
	"os"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/feeds/clients/edge"
	"code.justin.tv/feeds/clients/shine"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/populate-rec-feed/cmd/populate-rec-feed/internal/populatefeed"
	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

var instance = utility{}

type args struct {
	feedEntity entity.Entity
	entities   []entity.Entity
}

type utility struct {
	utilityCommon populatefeed.UtilityCommon
	configs       struct {
		DyanamoConfig   populatefeed.StorageConfig
		shineConfig     shine.Config
		feedsEdgeConfig feedsEdgeConfig
	}
	feedStorage     *populatefeed.StoryFeedStorage
	log             log.ElevatedLog
	shineClient     *shine.Client
	feedsEdgeClient edge.Client
}

func (f *utility) Main() {
	ctx := context.Background()

	if err := f.setup(); err != nil {
		f.utilityCommon.Log.Log("err", err, "unable to load initial config")
		os.Exit(1)
	}

	if err := f.inject(); err != nil {
		f.utilityCommon.Log.Log("err", err, "unable to inject dependencies")
		os.Exit(1)
	}

	args, err := f.parseArgs()
	if err != nil {
		f.utilityCommon.Log.Log("err", err, "parsing args failed")
		os.Exit(1)
	}

	activities, err := f.populateActivities(ctx, args.entities)
	if err != nil {
		f.utilityCommon.Log.Log("err", err, "populating activities failed")
		os.Exit(1)
	}

	err = f.addActivitiesToFeed(ctx, args.feedEntity, activities)
	if err != nil {
		f.utilityCommon.Log.Log("err", err, "adding activities to feed failed")
		os.Exit(1)
	}
}

func (f *utility) setup() error {

	if err := f.utilityCommon.Setup(&populatefeed.UtilityConfig{
		Team:        "feeds",
		UtilityName: "populate-rec-feed",
	}); err != nil {
		return err
	}

	distConf := f.utilityCommon.DistConfig
	if err := f.configs.DyanamoConfig.Load(distConf); err != nil {
		return err
	}
	if err := f.configs.shineConfig.Load(distConf); err != nil {
		return err
	}
	if err := f.configs.feedsEdgeConfig.Load(distConf); err != nil {
		return err
	}

	return nil
}

func (f *utility) inject() error {
	session, awsConf := service_common.CreateAWSSession(f.utilityCommon.DistConfig)
	dynamoConfig := append([]*aws.Config{
		{
			MaxRetries: aws.Int(10),
		}},
		awsConf...)

	f.feedStorage = &populatefeed.StoryFeedStorage{
		Config: &f.configs.DyanamoConfig,
		Dynamo: dynamodb.New(session, dynamoConfig...),
		Log:    f.utilityCommon.Log,
	}

	f.shineClient = &shine.Client{
		Config: &f.configs.shineConfig,
	}

	var err error
	f.feedsEdgeClient, err = edge.NewClient(twitchhttp.ClientConf{
		Host: f.configs.feedsEdgeConfig.host.Get(),
	})

	return err
}

func (f *utility) parseArgs() (*args, error) {
	feedEntityStr := flag.String("feedEntity", "", "A feed entity, e.g. r:47248162")
	rawEntitiesStr := flag.String("entities", "", "A comma separated list of entities")
	flag.Parse()

	if feedEntityStr == nil || *feedEntityStr == "" {
		return nil, errors.New("no feed entity given")
	}
	feedEntity, err := entity.Decode(*feedEntityStr)
	if err != nil {
		return nil, errors.Wrapf(err, "decoding string \"%s\" to entity failed", feedEntityStr)
	}
	ns := feedEntity.Namespace()
	if ns != "c" && ns != "n" && ns != "r" {
		return nil, errors.Errorf("given feed entity \"%s\" was not a channel feed, newsfeed or recommended feed entity", feedEntityStr)
	}

	userID := feedEntity.ID()
	if _, err := strconv.Atoi(userID); err != nil {
		return nil, errors.Errorf("the ID component of the feed entity is expected to be a number")
	}

	if rawEntitiesStr == nil || *rawEntitiesStr == "" {
		return nil, errors.New("a comma separated list of entities is required")
	}

	entityStrings := strings.Split(*rawEntitiesStr, ",")
	entities := make([]entity.Entity, 0, len(entityStrings))
	for _, entityString := range entityStrings {
		entityString = strings.TrimSpace(entityString)
		ent, err := entity.Decode(entityString)
		if err != nil {
			return nil, errors.Wrapf(err, "decoding string \"%s\" to entity failed", entityString)
		}

		ns := ent.Namespace()
		if ns != entity.NamespaceBroadcast &&
			ns != entity.NamespaceVod &&
			ns != entity.NamespaceClip &&
			ns != entity.NamespacePost {
			return nil, errors.Errorf("given entity \"%s\" was not a post, VOD, clip, or broadcast entity", entityString)
		}

		entities = append(entities, ent)
	}

	return &args{
		feedEntity: feedEntity,
		entities:   entities,
	}, nil
}

func (f *utility) populateActivities(ctx context.Context, entities []entity.Entity) ([]populatefeed.Activity, error) {
	activities := make([]populatefeed.Activity, 0, len(entities))
	for _, ent := range entities {

		userID := ""

		ns := ent.Namespace()
		if ns == entity.NamespacePost {
			post, err := f.feedsEdgeClient.GetPost(ctx, ent.ID(), nil, nil)
			if err != nil {
				return nil, errors.Wrapf(err, "getting post with ID, \"%s\", failed", ent.ID())
			}

			userID = post.UserID
		} else if ns == entity.NamespaceBroadcast {
			userIDAndBroadcastID := strings.Split(ent.ID(), "-")
			if len(userIDAndBroadcastID) != 2 {
				return nil, errors.Errorf("broadcast entity, \"%s\", is malformed", ent.Encode())
			}

			userID = userIDAndBroadcastID[0]
		} else {
			embed, err := f.shineClient.GetEmbedForEntity(ctx, ent, nil)
			if err != nil {
				return nil, errors.Wrapf(err, "getting embed for entity, \"%s\" failed", ent.Encode())
			}
			userID = embed.AuthorID
		}

		if userID == "" {
			return nil, errors.Errorf("author ID for entity, \"%s\", could not be found", ent.Encode())
		}

		activity := populatefeed.Activity{
			Entity: ent,
			Verb:   verb.Create,
			Actor:  entity.New(entity.NamespaceUser, userID),
		}
		activities = append(activities, activity)

		f.utilityCommon.Log.Log(
			"entity", ent.Encode(),
			"actor", userID,
		)
	}

	return activities, nil
}

func (f *utility) addActivitiesToFeed(ctx context.Context, feedEntity entity.Entity, activities []populatefeed.Activity) error {

	feedEntityStr := feedEntity.Encode()

	for _, activity := range activities {
		storyID := activity.Entity.Encode()
		score := scoreForTime(time.Now())
		if err := f.feedStorage.SaveStory(ctx, feedEntityStr, storyID, activity, score); err != nil {
			return err
		}

		time.Sleep(200 * time.Millisecond)
	}

	return nil
}

type feedsEdgeConfig struct {
	host *distconf.Str
}

func (s *feedsEdgeConfig) Load(dconf *distconf.Distconf) error {
	s.host = dconf.Str("feeds-edge.http_endpoint", "")
	if s.host.Get() == "" {
		return errors.New("unable to find feeds-edge addresss (feeds-edge.http_endpoint)")
	}
	return nil
}

func scoreForTime(t time.Time) float64 {
	return t.Sub(time.Unix(0, 0)).Hours()
}

func main() {
	instance.Main()
}
