package shine

import (
	"fmt"
	"net/http"
	"net/url"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"github.com/google/go-querystring/query"
	"golang.org/x/net/context"
)

// Embed contains information related to the RequestURL
type Embed struct {
	RequestURL      string      `json:"request_url"`
	Type            string      `json:"type"`
	Title           string      `json:"title"`
	Description     string      `json:"description"`
	AuthorName      string      `json:"author_name"`
	AuthorURL       string      `json:"author_url,omitempty"`
	AuthorThumbnail string      `json:"author_thumbnail_url,omitempty"`
	AuthorID        string      `json:"author_id,omitempty"`
	ThumbnailURL    string      `json:"thumbnail_url"`
	Thumbnails      *Thumbnails `json:"thumbnail_urls,omitempty"`
	PlayerHTML      string      `json:"html"`
	ProviderName    string      `json:"provider_name"`

	// Twitch VOD/Clips specific
	CreatedAt       *time.Time `json:"created_at"`
	Game            string     `json:"game"`
	VideoLength     int        `json:"video_length"`
	ViewCount       int        `json:"view_count,omitempty"`
	TwitchType      string     `json:"twitch_type"`
	TwitchContentID string     `json:"twitch_content_id,omitempty"`
	StartTime       *time.Time `json:"start_time"`
	EndTIme         *time.Time `json:"end_time"`
}

// Thumbnails provides additional thumbnail URLs
type Thumbnails struct {
	Large  string `json:"large,omitempty"`
	Medium string `json:"medium,omitempty"`
	Small  string `json:"small,omitempty"`
	Tiny   string `json:"tiny,omitempty"`
}

// Config configures shine http client
type Config struct {
	Host func() string
}

// Load configuration information
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.Host = dconf.Str("shine.http_endpoint", "").Get
	return nil
}

// Client is used by third parties to get embed information from shine
type Client struct {
	RequestDoer    clients.RequestDoer
	NewHTTPRequest clients.NewHTTPRequest
	Config         *Config
}

type errResponse struct {
	StatusCode int    `json:"-"`
	Message    string `json:"message"`
}

func (e errResponse) Error() string {
	return fmt.Sprintf("%d: %s", e.StatusCode, e.Message)
}

func (c *Client) http(ctx context.Context, method string, path string, queryParams url.Values, body interface{}, into interface{}) error {
	return clients.DoHTTP(ctx, c.RequestDoer, method, c.Config.Host()+path, queryParams, body, into, c.NewHTTPRequest)
}

// GetEmbedOptions specifies the optional parameters of the GetEmbed function
type GetEmbedOptions struct {
	Autoplay *bool `url:"autoplay"`
}

// GetEmbed grabs embed information for a provided URL
func (c *Client) GetEmbed(ctx context.Context, embedURL string, options *GetEmbedOptions) (*Embed, error) {
	var response Embed
	path := "/v1/embed"
	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("url", embedURL)

	if err := c.http(ctx, "GET", path, query, nil, &response); err != nil {
		if clients.ErrorCode(err) == http.StatusUnprocessableEntity {
			return nil, nil
		}
		return nil, err
	}
	return &response, nil
}

// GetEmbedForEntity returns embed information for the given entity
func (c *Client) GetEmbedForEntity(ctx context.Context, ent entity.Entity, options *GetEmbedOptions) (*Embed, error) {
	var response Embed
	path := "/v1/embed_for_entity"
	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("entity", ent.Encode())

	if err := c.http(ctx, "GET", path, query, nil, &response); err != nil {
		if clients.ErrorCode(err) == http.StatusUnprocessableEntity {
			return nil, nil
		}
		return nil, err
	}
	return &response, nil
}
