package sandyconf

import (
	"code.justin.tv/systems/sandstorm/manager"
	"time"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/credentials"
)

const (
	defaultTableName = "sandstorm-production"
	defaultKeyID = "alias/sandstorm-production"
	defaultDatacenter = "us-west-2"
)

// ManagerConstructor provides an easy way to get a sandstorm manager with reasonable defaults.
type ManagerConstructor struct {
	AwsConfig *aws.Config
	TableName string
	KeyID string
}

func (m *ManagerConstructor) tableName() string {
	if m.TableName == "" {
		return defaultTableName
	}
	return m.TableName
}

func (m *ManagerConstructor) keyID() string {
	if m.KeyID == "" {
		return defaultKeyID
	}
	return m.KeyID
}

func (m *ManagerConstructor) region() *string {
	if m.AwsConfig == nil || m.AwsConfig.Region == nil {
		return aws.String(defaultDatacenter)
	}
	return m.AwsConfig.Region
}

// Region sets the region to use for the constructor
func (m *ManagerConstructor) Region(region string) *ManagerConstructor{
	if m.AwsConfig == nil {
		m.AwsConfig = &aws.Config{}
	}
	m.AwsConfig.Region = &region
	return m
}

// Manager creates a manager that assumes an ARN role
func (m *ManagerConstructor) Manager(roleARN string) *manager.Manager {
	awsConfig := &aws.Config{}
	awsConfig.MergeIn(m.AwsConfig)
	awsConfig.Region = m.region()
	stsclient := sts.New(session.New(m.AwsConfig))
	arp := &stscreds.AssumeRoleProvider{
		ExpiryWindow: 10 * time.Second,
		RoleARN:      roleARN,
		Client:       stsclient,
	}
	credentials := credentials.NewCredentials(arp)
	awsConfig.WithCredentials(credentials)
	mgr := manager.New(manager.Config{
		AWSConfig: awsConfig,
		TableName: m.tableName(),
		KeyID:     m.keyID(),
	})
	return mgr
}