package fmtlogger

import (
	"io"
	"io/ioutil"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"github.com/go-logfmt/logfmt"
)

// LogfmtLogger logs out in logfmt format
type LogfmtLogger struct {
	Out             io.Writer
	MissingValueKey log.Key
}

// NewLogfmtLogger returns a logger that encodes keyvals to the Writer in
// logfmt format. The passed Writer must be safe for concurrent use by
// multiple goroutines if the returned Logger will be used concurrently.
func NewLogfmtLogger(w io.Writer, ErrHandler log.ErrorHandler) log.Logger {
	if w == ioutil.Discard {
		return log.Discard
	}
	return &log.ErrorLogLogger{
		RootLogger: &LogfmtLogger{
			Out:             w,
			MissingValueKey: log.Msg,
		},
		ErrHandler: ErrHandler,
	}
}

// Log writes the keyvalus in logfmt format to Out
func (l *LogfmtLogger) Log(keyvals ...interface{}) error {
	// The Logger interface requires implementations to be safe for concurrent
	// use by multiple goroutines. For this implementation that means making
	// only one call to l.w.Write() for each call to Log. We first collect all
	// of the bytes into b, and then call l.w.Write(b).
	if len(keyvals)%2 != 0 {
		keyvals = append(keyvals[0:len(keyvals)-1:len(keyvals)-1], l.MissingValueKey, keyvals[len(keyvals)-1])
	}
	b, err := logfmt.MarshalKeyvals(keyvals...)
	if err != nil {
		return err
	}
	b = append(b, '\n')
	if _, err := l.Out.Write(b); err != nil {
		return errors.Wrap(err, "cannot write out logfmt for log")
	}
	return nil
}
