package sfximporter

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"runtime/debug"
	"strconv"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"github.com/google/go-github/github"
	"github.com/signalfx/golib/datapoint/dpsink"
	"github.com/signalfx/golib/event"
	"golang.org/x/net/context"
)

type CleanDeployEventImporterConfig struct {
	PollInterval *distconf.Duration
}

func (c *CleanDeployEventImporterConfig) Load(d *distconf.Distconf) error {
	c.PollInterval = d.Duration("sfximporter.clean_deploy_interval", time.Second*10)
	return nil
}

type CleanDeployEventImporter struct {
	Client  *CleanDeployClient
	Config  *CleanDeployEventImporterConfig
	SfxSink dpsink.Sink
	Log     log.Logger
	tracker deploymentTracker

	closeChan chan struct{}
}

func (c *CleanDeployEventImporter) Setup() error {
	c.closeChan = make(chan struct{})
	c.tracker.log = c.Log
	return nil
}

func (c *CleanDeployEventImporter) Start() error {
	defer func() {
		if r2 := recover(); r2 != nil {
			debug.PrintStack()
			panic(r2)
		}
	}()
	for {
		select {
		case <-c.closeChan:
			return nil
		case <-time.After(c.Config.PollInterval.Get()):
		}
		c.processEvents()
	}
}

func (c *CleanDeployEventImporter) processEvents() {
	deploymentCountToFetch := 30
	deps, err := c.Client.RecentDeploys(&RecentDeploysArgs{
		Count: &deploymentCountToFetch,
	})
	if err != nil {
		c.Log.Log("err", err, "unable to get deployments")
		return
	}
	newDeployments := c.tracker.feedDeployments(*deps)
	if len(newDeployments) == 0 {
		//c.Log.Log("No new deployments")
		return
	}
	events := make([]*event.Event, 0, len(newDeployments))
	for _, dep := range newDeployments {
		event := toSfxEvent(dep)
		if event != nil {
			events = append(events, event)
			c.Log.Log("event", event, "new event!")
		}
	}
	ctx, onDone := context.WithTimeout(context.Background(), time.Second)
	defer onDone()
	if err := c.SfxSink.AddEvents(ctx, events); err != nil {
		c.Log.Log("err", err, "unable to send sfx events")
	}
	c.tracker.updateLastDeploymentID(newDeployments)
}

type deploymentTracker struct {
	lastDeploymentID int64
	log              log.Logger
}

func (d *deploymentTracker) feedDeployments(in []Deployment) []Deployment {
	in = filterUnprocessable(in)
	if len(in) == 0 {
		return nil
	}
	if d.lastDeploymentID == 0 {
		d.lastDeploymentID = *in[0].ID
		d.log.Log("id", d.lastDeploymentID, "setting deployment ID")
		return nil
	}
	newDeployments := make([]Deployment, 0, len(in))
	for _, dep := range in {
		if *dep.ID > d.lastDeploymentID {
			newDeployments = append(newDeployments, dep)
		} else {
			break
		}
	}
	//	d.log.Log("count", len(newDeployments), "New deployment count")
	return newDeployments
}

func (d *deploymentTracker) updateLastDeploymentID(deployments []Deployment) {
	for _, dep := range deployments {
		if d.lastDeploymentID < *dep.ID {
			d.lastDeploymentID = *dep.ID
		}
	}
}

func filterUnprocessable(in []Deployment) []Deployment {
	ret := make([]Deployment, 0, len(in))
	for _, d := range in {
		if d.ID == nil {
			continue
		}
		ret = append(ret, d)
	}
	return ret
}

func getSfxDims(in Deployment) (map[string]string, string) {
	dims := map[string]string{
		"source": "clean_deploy",
	}
	metricName := "code_deployment"
	if in.Environment != nil {
		dims["environment"] = *in.Environment
	}
	if in.Owner != nil {
		dims["owner"] = *in.Owner
		metricName += "-" + *in.Owner
	}
	if in.Repository != nil {
		dims["repository"] = *in.Repository
		metricName += "-" + *in.Repository
	}
	return dims, metricName
}

func getSfxProperties(in Deployment) map[string]interface{} {
	properties := map[string]interface{}{}
	if in.Branch != nil {
		properties["branch"] = *in.Branch
	}
	if in.Creator != nil {
		properties["creator"] = *in.Creator
	}
	if in.Description != nil {
		properties["description"] = *in.Description
	}
	if in.JenkinsJob != nil {
		properties["jenkins_job"] = *in.JenkinsJob
	}
	if in.SHA != nil {
		properties["SHA"] = *in.SHA
	}
	dlink := directLink(in)
	if dlink != "" {
		properties["job_link"] = dlink
	}
	return properties
}

func toSfxEvent(in Deployment) *event.Event {
	dims, metricName := getSfxDims(in)
	properties := getSfxProperties(in)

	createdAt := time.Now()
	if in.CreatedAt != nil {
		createdAt = *in.CreatedAt
	}
	return event.NewWithProperties(metricName, event.USERDEFINED, dims, properties, createdAt)
}

func directLink(in Deployment) string {
	if in.Owner == nil || in.Repository == nil || in.ID == nil {
		return ""
	}
	return fmt.Sprintf("https://clean-deploy.internal.justin.tv/#/%s/%s/deploys/%d", *in.Owner, *in.Repository, *in.ID)
}

func (c *CleanDeployEventImporter) Close() error {
	close(c.closeChan)
	return nil
}

type CleanDeployClientConfig struct {
	GithubAuthToken *distconf.Str
	Endpoint        *distconf.Str
}

func (c *CleanDeployClientConfig) Load(d *distconf.Distconf) error {
	c.Endpoint = d.Str("sfximporter.clean_deploy_url", "https://clean-deploy.internal.justin.tv")
	return nil
}

func (c *CleanDeployClientConfig) LoadSecrets(d *distconf.Distconf) error {
	c.GithubAuthToken = d.Str("sfximporter.github_auth_token", "")
	if c.GithubAuthToken.Get() == "" {
		return errors.New("unable to find sfximporter.github_auth_token")
	}
	return nil
}

type CleanDeployClient struct {
	Client http.Client
	Config *CleanDeployClientConfig
	Log    log.Logger
}

type RecentDeploysArgs struct {
	Count *int
}

type RecentDeploysResponse []Deployment

// Deployment represents a rollout of an application change
type Deployment struct {
	ID            *int64  `json:"id,omitempty" db:"id"`
	GithubID      *int    `json:"github_id,omitempty" db:"githubid"`
	CodeDeployID  *[]byte `json:"codedeploy_id,omitempty" db:"codedeployid"`
	Repository    *string `json:"repository,omitempty" db:"repository"`
	Owner         *string `json:"owner,omitempty" db:"owner"`
	SHA           *string `json:"sha,omitempty" db:"sha"`
	Branch        *string `json:"branch,omitempty" db:"branch"`
	Environment   *string `json:"environment,omitempty" db:"environment"`
	Creator       *string `json:"creator,omitempty" db:"creator"`
	State         *string `json:"state,omitempty" db:"state"`
	Description   *string `json:"description,omitempty" db:"description"`
	PreviousSHA   *string `json:"previous_sha,omitempty" db:"previoussha"`
	CodeReviewURL *string `json:"code_review_url,omitempty" db:"codereviewurl"`
	Severity      *string `json:"severity,omitempty" db:"severity"`
	Hosts         *string `json:"hosts,omitempty" db:"hosts"`
	JenkinsJob    *string `json:"jenkinsjob,omitempty" db:"jenkinsjob"`
	Link          *int64  `json:"link,omitempty" db:"link"`

	GithubCreator     *github.User              `json:"github_creator"`
	CommitsComparison *github.CommitsComparison `json:"commits_comparison"`

	CreatedAt *time.Time `json:"created_at,omitempty" db:"createdat"`
	UpdatedAt *time.Time `json:"updated_at,omitempty" db:"updatedat"`
}

func (c *CleanDeployClient) RecentDeploys(args *RecentDeploysArgs) (*RecentDeploysResponse, error) {
	if args == nil {
		args = &RecentDeploysArgs{}
	}
	req, err := http.NewRequest(http.MethodGet, c.Config.Endpoint.Get()+"/v1/deployments", nil)
	if err != nil {
		return nil, err
	}
	req.Header.Add("GithubAccessToken", c.Config.GithubAuthToken.Get())
	queryParams := make(url.Values)
	if args.Count != nil {
		queryParams.Set("count", strconv.Itoa(*args.Count))
	}
	req.URL.RawQuery = queryParams.Encode()
	resp, err := c.Client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
			c.Log.Log("err", err, "unable to close response body")
		}
	}()
	if resp.StatusCode != http.StatusOK {
		return nil, errors.Errorf("Invalid status code: %d", resp.StatusCode)
	}
	ret := make(RecentDeploysResponse, 0, 10)
	if err := json.NewDecoder(resp.Body).Decode(&ret); err != nil {
		return nil, err
	}
	return &ret, nil
}
