package main

import (
	"net"
	"os"
	"os/signal"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/sfximporter/cmd/sfximporter/internal/sfximporter"
	"github.com/signalfx/golib/datapoint/dpsink"
)

// CodeVersion is set by build script
var CodeVersion string

const (
	teamName    = "feeds"
	serviceName = "sfximporter"
)

func sfxStastdConfig() sfxstatsd.SetupConfig {
	return sfxstatsd.SetupConfig{}
}

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion:    CodeVersion,
		SfxSetupConfig: sfxStastdConfig(),
	},
}

type config struct {
	DryRun *distconf.Bool
}

func (c *config) Load(d *distconf.Distconf) error {
	c.DryRun = d.Bool("sfximporter.dryrun", true)
	return nil
}

type service struct {
	osExit        func(code int)
	Runner        service_common.ServiceRunner
	serviceCommon service_common.ServiceCommon
	sigChan       chan os.Signal
	onListen      func(listeningAddr net.Addr)

	server      sfximporter.HTTPServer
	cleanDeploy sfximporter.CleanDeployEventImporter

	configs struct {
		config                  config
		httpConfig              sfximporter.HTTPConfig
		cleanDeployClientConfig sfximporter.CleanDeployClientConfig
		cleanDeployConfig       sfximporter.CleanDeployEventImporterConfig
	}
}

func (s *service) setup() error {
	if err := s.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(
		s.serviceCommon.Config,
		&s.configs.httpConfig,
		&s.configs.config,
		&s.configs.cleanDeployConfig,
		&s.configs.cleanDeployClientConfig,
	); err != nil {
		return err
	}

	if err := s.configs.cleanDeployClientConfig.LoadSecrets(s.serviceCommon.Secrets); err != nil {
		return err
	}
	return nil
}

func (s *service) inject() {
	s.server = sfximporter.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Log:         s.serviceCommon.Log,
			Config:      &s.configs.httpConfig.BaseHTTPServerConfig,
			ElevateKey:  s.serviceCommon.ElevateLogKey,
			Ctxlog:      &s.serviceCommon.Ctxlog,
			Dims:        &s.serviceCommon.CtxDimensions,
			OnListen:    s.onListen,
			PanicLogger: s.serviceCommon.PanicLogger,
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			XRay: s.serviceCommon.XRay,
		},
	}
	s.server.BaseHTTPServer.SetupRoutes = s.server.SetupRoutes

	cdClient := sfximporter.CleanDeployClient{
		Config: &s.configs.cleanDeployClientConfig,
		Log:    s.serviceCommon.Log,
	}

	sfxClient := s.serviceCommon.SfxResult.Scheduler.Sink.(dpsink.Sink)
	if s.configs.config.DryRun.Get() {
		sfxClient = &sfximporter.SfxLoggedClient{
			Log: s.serviceCommon.Log,
		}
	}

	s.cleanDeploy = sfximporter.CleanDeployEventImporter{
		Log:     s.serviceCommon.Log,
		Config:  &s.configs.cleanDeployConfig,
		Client:  &cdClient,
		SfxSink: sfxClient,
	}
	s.Runner = service_common.ServiceRunner{
		Log: s.serviceCommon.Log,
		Services: []service_common.Service{
			&s.server,
			&s.serviceCommon,
			&s.cleanDeploy,
		},
		SigChan:      s.sigChan,
		SignalNotify: signal.Notify,
	}
	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"aws-sdk-go", "net/http"},
	}).Check(s, s.Runner)
	res.MustBeEmpty(os.Stderr)
}

func (s *service) main() {
	if err := s.setup(); err != nil {
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		s.osExit(1)
		return
	}
	s.inject()
	if err := s.Runner.Execute(); err != nil {
		service_common.SetupLogger.Log("err", err, "wait to end finished with an error")
		s.osExit(1)
		return
	}
	s.serviceCommon.Log.Log("done with service")
}

func main() {
	instance.main()
}
