locals {
  owner   = "jlindamo@twitch.tv"
  team    = "feeds"
  region  = "us-west-2"
  service = "sfximporter"

  account_ids = {
    staging    = "724951484461"
    production = "914569885343"
  }

  profiles = {
    staging    = "twitch-feed-dev"
    production = "twitch-feed-aws"
  }

  min_counts = {
    staging    = "1"
    production = "1"
  }

  desired_counts = {
    staging    = "1"
    production = "1"
  }

  max_counts = {
    staging    = "1"
    production = "1"
  }

  auto_promotes = {
    staging    = "true"
    production = "false"
  }

  promote_froms = {
    staging    = "latest"
    production = "staging"
  }

  promote_from  = "${lookup(local.promote_froms, terraform.workspace)}"
  auto_promote  = "${lookup(local.auto_promotes, terraform.workspace)}"
  min_count     = "${lookup(local.min_counts, terraform.workspace)}"
  max_count     = "${lookup(local.max_counts, terraform.workspace)}"
  desired_count = "${lookup(local.desired_counts, terraform.workspace)}"
  profile       = "${lookup(local.profiles, terraform.workspace)}"
}

provider "aws" {
  profile             = "${local.profile}"
  allowed_account_ids = ["${lookup(local.account_ids, terraform.workspace)}"]
  region              = "${local.region}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "feeds/sfximporter-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

provider "jenkins" {
  url                = "https://jenkins.internal.justin.tv"
  shared_credentials = "${file(pathexpand("~/.jenkins_auth"))}"
}

provider "consul" {
  address    = "consul.internal.justin.tv"
  datacenter = "us-west2"
}

data "terraform_remote_state" "account" {
  backend = "s3"

  config {
    bucket  = "${local.profile}"
    key     = "tfstate/feed/terraform/accounts/${local.profile}"
    region  = "us-west-2"
    profile = "${local.profile}"
  }
}

data "terraform_remote_state" "account_core" {
  backend     = "s3"
  environment = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-account.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "region_core" {
  backend     = "s3"
  environment = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-region-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

module "sfximporter" {
  source           = "../../modules/sfximporter/cluster"
  environment      = "${terraform.workspace}"
  target_group_arn = "${module.alb.target_group_arn}"
  cluster_name     = "${data.terraform_remote_state.region_core.common_cluster_name}"

  min_size                  = "${local.min_count}"
  desired_count             = "${local.desired_count}"
  max_size                  = "${local.max_count}"
  service_iam_role          = "${data.terraform_remote_state.account_core.service_iam_role}"
  task_autoscale_iam_role   = "${data.terraform_remote_state.account_core.task_autoscale_iam_role}"
  profile                   = "${local.profile}"
  promote_from              = "${local.promote_from}"
  auto_promote              = "${local.auto_promote}"
  deploy_aws_role           = "${data.terraform_remote_state.account_core.build_role}"
  container_loggroup_region = "${data.terraform_remote_state.region_core.container_loggroup_region}"
  container_loggroup        = "${data.terraform_remote_state.region_core.container_loggroup}"
  aws_creds                 = "${data.terraform_remote_state.account_core.jenkins_build_secret}"
}

module "permissions" {
  source      = "../../modules/sfximporter/permissions"
  environment = "${terraform.workspace}"
  iam_id      = "${module.sfximporter.iam_id}"
}

module "alb" {
  source            = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//alb?ref=v0.8.11"
  owner             = "${local.owner}"
  environment       = "${terraform.workspace}"
  access_log_bucket = "${local.profile}-logs"
  team              = "${local.team}"
  service           = "${local.service}"
  subnets           = "${data.terraform_remote_state.account.private_subnets}"
  security_groups   = "${data.terraform_remote_state.account.twitch_subnets_sg}"
  vpc_id            = "${data.terraform_remote_state.account.vpc_id}"
}

output target_group_arn {
  value = "${module.alb.target_group_arn}"
}
