package api

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"sync"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedcache"
	"code.justin.tv/feeds/spade"
	"goji.io"
	"goji.io/pat"
)

// HTTPConfig configures a HTTPServer
type HTTPConfig struct {
	service_common.BaseHTTPServerConfig
	embedAutoplayDefault *distconf.Bool
	defaultEmbedTTL      *distconf.Duration
	streamTTL            *distconf.Duration
	eventTTL             *distconf.Duration
	vodTTL               *distconf.Duration
	oembedTimeout        *distconf.Duration
}

// Load config from distconf
func (s *HTTPConfig) Load(dconf *distconf.Distconf) error {
	if err := s.Verify(dconf, "shine"); err != nil {
		return err
	}
	s.embedAutoplayDefault = dconf.Bool("embed_autoplay_default", true)
	s.defaultEmbedTTL = dconf.Duration("shine.ttl", time.Minute*60)
	s.streamTTL = dconf.Duration("shine.stream_ttl", time.Minute*1)
	s.eventTTL = dconf.Duration("shine.event_ttl", time.Minute*15)
	s.vodTTL = dconf.Duration("shine.vod_ttl", time.Minute*15)
	s.oembedTimeout = dconf.Duration("shine.oembed_timeout", time.Millisecond*500)
	return nil
}

// HTTPServer answers masonry HTTP requests
type HTTPServer struct {
	service_common.BaseHTTPServer
	Config           *HTTPConfig
	Redis            *feedcache.ObjectCache
	SpadeClient      *spade.Client
	DefaultProviders []Provider
	TwitchProviders  []Provider
}

func (s *HTTPServer) createHandler(name string, callback func(req *http.Request) (interface{}, error)) *service_common.JSONHandler {
	return &service_common.JSONHandler{
		Log:          s.Log,
		Stats:        s.Stats.NewSubStatSender(name),
		ItemCallback: callback,
	}
}

// SetupRoutes configures shine goji routes for shine
func (s *HTTPServer) SetupRoutes(mux *goji.Mux) {
	mux.Handle(pat.Get("/v1/embed_for_url"), s.createHandler("get_embed_for_url", s.getEmbedForURL))
	mux.Handle(pat.Get("/v1/embed_for_entity"), s.createHandler("get_embed_for_entity", s.getEmbedForEntity))

	mux.Handle(pat.Post("/v1/oembeds_for_urls"), s.createHandler("get_oembeds_for_urls", s.getOembedsForURLs))
	mux.Handle(pat.Post("/v1/entities_for_urls"), s.createHandler("get_entities_for_urls", s.getEntitiesForURLs))
}

func (s *HTTPServer) getEmbedForURL(req *http.Request) (interface{}, error) {
	var err error
	urlParam := req.URL.Query().Get("url")

	autoplay, err := s.getWhetherAutoplay(req)
	if err != nil {
		return nil, err
	}

	formattedURL, err := formatURL(urlParam)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "malformed url"),
		}
	}

	embed, embedErr := s.getEmbed(req.Context(), formattedURL, autoplay, s.getProviders(req))
	if embedErr != nil {
		return nil, embedErr
	}
	if embed == nil || (embed.RequestURL == "" && embed.TwitchType != "live") {
		return nil, &service_common.CodedError{
			Code: http.StatusUnprocessableEntity,
			Err:  errors.New("request url is not supported"),
		}
	}
	return *embed, nil
}

func (s *HTTPServer) getEmbedForEntity(req *http.Request) (interface{}, error) {
	autoplay, err := s.getWhetherAutoplay(req)
	if err != nil {
		return nil, err
	}

	entityString := req.URL.Query().Get("entity")
	if entityString == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("no entity provided"),
		}
	}

	ent, err := entity.Decode(entityString)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "given entity could not be decoded"),
		}
	}

	url, err := entity.ToURL(ent)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusUnprocessableEntity,
			Err:  errors.Wrap(err, "given entity is not supported"),
		}
	}

	return s.getEmbed(req.Context(), url, autoplay, s.getProviders(req))
}

type getOembedsForURLsParams struct {
	URLs []string `json:"urls"`
}

// dedupeURLs filters out duplicate instances of an url from urls
func dedupeURLs(urls []string) []string {
	var uniqueURLs []string
	seen := map[string]bool{}

	for _, url := range urls {
		if !seen[url] {
			seen[url] = true
			uniqueURLs = append(uniqueURLs, url)
		}
	}
	return uniqueURLs
}

// getOembedsForURLs expects a POST request, whose body is a slice of urls.
// getOemebdsForURLs resolves each url into the appropriate Oembed structure.
//
// NOTE: malformed urls are dropped silently and are not part of the response body
// NOTE: getOembedsForURLs enforces an upper limit of 100 urls per request.
func (s *HTTPServer) getOembedsForURLs(req *http.Request) (interface{}, error) {
	var bulkURLs getOembedsForURLsParams
	dec := json.NewDecoder(req.Body)
	if err := dec.Decode(&bulkURLs); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "error decoding request body"),
		}
	}

	urls := bulkURLs.URLs
	if len(urls) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide at least one url"),
		}
	}

	const maxGetOembeds = 100
	urls = dedupeURLs(urls)
	if len(urls) > maxGetOembeds {
		return nil, &service_common.CodedError{
			Code: http.StatusRequestEntityTooLarge,
			Err:  fmt.Errorf("server can only handle %d urls per request", maxGetOembeds),
		}
	}

	var wg sync.WaitGroup
	var mu sync.Mutex
	response := &OembedsForURLs{}
	ctx := req.Context()

	for _, url := range urls {
		wg.Add(1)

		go func(url string) {
			defer wg.Done()

			formattedURL, err := formatURL(url)
			if err != nil {
				return
			}

			timeoutCtx, cancel := context.WithTimeout(ctx, s.Config.oembedTimeout.Get())
			defer cancel()

			oembed, err := s.getOembed(timeoutCtx, formattedURL, false, s.getProviders(req))
			if err != nil {
				s.Log.LogCtx(ctx, "url", url, "formattedURL", formattedURL, "error", err)
				return
			}
			if oembed == nil {
				return
			}

			mu.Lock()
			response.Oembeds = append(response.Oembeds, URLAndOembed{URL: url, Oembed: oembed})
			mu.Unlock()
		}(url)
	}

	wg.Wait()
	return response, nil
}

// getEntitiesFromURLsParams contains a slice of URLs passed in to /v1/entities_for_bulk_urls
type getEntitiesFromURLsParams struct {
	URLs []string `json:"urls"`
}

// getEntitiesForURLs expects a POST request, whose body is a slice of urls.
// getEntitiesForURLs resolves each url into the appropriate entity.
func (s *HTTPServer) getEntitiesForURLs(req *http.Request) (interface{}, error) {
	var bulkURLs getEntitiesFromURLsParams
	dec := json.NewDecoder(req.Body)
	if err := dec.Decode(&bulkURLs); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "error decoding request body"),
		}
	}

	urls := bulkURLs.URLs
	if len(urls) == 0 {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("must provide at least one url"),
		}
	}

	providers := s.getProviders(req)

	response := &EntitiesForURLs{}
	for _, url := range urls {
		for _, p := range providers {
			if en, err := p.EntityForURL(url); err == nil {
				ue := URLAndEntity{
					URL:    url,
					Entity: en,
				}
				response.Entities = append(response.Entities, ue)
				break
			}
		}
	}
	return response, nil
}

func (s *HTTPServer) getWhetherAutoplay(req *http.Request) (bool, error) {
	paramName := "autoplay"
	autoplayParam := req.URL.Query().Get(paramName)
	if autoplayParam == "" {
		return s.Config.embedAutoplayDefault.Get(), nil
	}

	autoplay, err := strconv.ParseBool(autoplayParam)
	if err != nil {
		return false, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, fmt.Sprintf("could not convert \"%s\" (%s query param) to bool", autoplayParam, paramName)),
		}
	}

	return autoplay, nil
}

// Attempts to standardize a URL. May not be necessary based on consumers' implementation of requests.
func formatURL(reqURL string) (string, error) {
	urlObj, err := url.Parse(reqURL)
	if err != nil {
		return "", err
	}

	// Sets URL scheme, otherwise HTTP GET requests for e.g. "www.youtube.com" fail
	if urlObj.Scheme == "" {
		urlObj.Scheme = "http"
	}
	// Allows some URLs with strange capitalization (like HTTP://www.YOUTUBE.com/foobar) to be handled correctly
	// NOTE: still somewhat incomplete handling, if the url is also missing scheme.
	// Results in negligible negative impact for caching
	urlObj.Host = strings.ToLower(urlObj.Host)
	urlObj.Scheme = strings.ToLower(urlObj.Scheme)
	return urlObj.String(), nil
}

func (s *HTTPServer) getProviders(req *http.Request) []Provider {
	providers := req.URL.Query().Get("providers")
	if strings.ToLower(providers) == "twitch" {
		return s.TwitchProviders
	}
	return s.DefaultProviders
}
