package api

import (
	"errors"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/spade"
	"golang.org/x/net/context"
)

// Provider is the interface for an embed provider
type Provider interface {
	Matches(url string) bool
	RequestEmbed(ctx context.Context, url string, autoplay bool) (*Embed, error)
	RequestOembed(ctx context.Context, url string, autplay bool) (*Oembed, error)
	EntityForURL(url string) (entity.Entity, error)
}

var ErrorURLDoesNotMatchProvider = errors.New("url does not match provider")

type codedError interface {
	HTTPCode() int
}

func (s *HTTPServer) getOembed(ctx context.Context, url string, autoplay bool, providers []Provider) (*Oembed, error) {
	start := time.Now()
	for _, provider := range providers {
		if provider.Matches(url) {
			var response Oembed
			key := "oembeds:" + url + "?autoplay=" + strconv.FormatBool(autoplay)
			err := s.Redis.Cached(ctx, key, func() (interface{}, error) {
				oembed, err := provider.RequestOembed(ctx, url, autoplay)
				if err != nil {
					return nil, err
				}
				if oembed == nil {
					return nil, nil
				}
				oembed.CacheAge = s.Config.defaultEmbedTTL.Get()
				s.recordEmbedStats(oembed.ProviderName, false, start)
				return *oembed, nil
			}, &response)
			if err != nil {
				s.Log.LogCtx(ctx,
					"oembedURL", url,
					"err", err,
				)
				return nil, err
			}
			return &response, nil
		}
	}
	return nil, nil
}

func (s *HTTPServer) getEmbed(ctx context.Context, url string, autoplay bool, providers []Provider) (*Embed, error) {
	requestStart := time.Now()
	for _, provider := range providers {
		if provider.Matches(url) {
			var response Embed
			key := "embeds:" + url + "?autoplay=" + strconv.FormatBool(autoplay)
			err := s.Redis.Cached(ctx, key, func() (interface{}, error) {
				embed, err := provider.RequestEmbed(ctx, url, autoplay)
				if err != nil {
					return nil, err
				}
				if embed == nil {
					return nil, nil
				}
				if embed.ProviderName == "Twitch" {
					s.trackEmbed(embed)
				}
				embed.defaultTTL = s.Config.defaultEmbedTTL.Get()
				embed.streamTTL = s.Config.streamTTL.Get()
				embed.eventTTL = s.Config.eventTTL.Get()
				embed.vodTTL = s.Config.eventTTL.Get()

				s.recordEmbedStats(embed.ProviderName, false, requestStart)
				return *embed, nil
			}, &response)
			if err != nil {
				s.Log.Log(
					"err", err,
					"embedURL", url,
				)
				return &Embed{RequestURL: url}, nil
			}
			s.Log.Debug("type", response.Type, "authorID", response.AuthorID)
			return &response, nil
		}
	}
	return nil, nil
}

func (s *HTTPServer) recordEmbedStats(providerName string, cached bool, requestStart time.Time) {
	if providerName == "" {
		providerName = "unknown_name"
	}
	prefix := "get_embed.stats." + strings.ToLower(providerName)
	if cached {
		prefix += ".cached"
	} else {
		prefix += ".uncached"
	}

	s.Stats.TimingDurationC(prefix+".duration", time.Since(requestStart), 1.0)
}

func (s *HTTPServer) trackEmbed(embed *Embed) {
	s.SpadeClient.QueueEvents(spade.Event{
		Name: "feed_server_embed",
		Properties: struct {
			RequestURL   string     `json:"request_url,omitempty"`
			Type         string     `json:"type,omitempty"`
			Title        string     `json:"title,omitempty"`
			Description  string     `json:"description,omitempty"`
			AuthorName   string     `json:"author_name,omitempty"`
			ThumbnailURL string     `json:"thumbnail_url,omitempty"`
			PlayerHTML   string     `json:"html,omitempty"`
			ProviderName string     `json:"provider_name,omitempty"`
			CreatedAt    *time.Time `json:"created_at,omitempty"`
			Game         string     `json:"game,omitempty"`
			VideoLength  int        `json:"video_length,omitempty"`
			TwitchType   string     `json:"twitch_type,omitempty"`
			StartTime    *time.Time `json:"start_time,omitempty"`
			EndTime      *time.Time `json:"end_time,omitempty"`
		}{
			RequestURL:   embed.RequestURL,
			Type:         embed.Type,
			Title:        embed.Title,
			Description:  embed.Description,
			AuthorName:   embed.AuthorName,
			ThumbnailURL: embed.ThumbnailURL,
			PlayerHTML:   embed.PlayerHTML,
			ProviderName: embed.ProviderName,
			CreatedAt:    embed.CreatedAt,
			Game:         embed.Game,
			VideoLength:  embed.VideoLength,
			TwitchType:   embed.TwitchType,
			StartTime:    embed.StartTime,
			EndTime:      embed.EndTime,
		},
	})
}
