package api

import (
	"time"

	"code.justin.tv/feeds/feeds-common/entity"
)

// Embed contains VOD embed information
type Embed struct {
	RequestURL      string      `json:"request_url,omitempty"`
	Type            string      `json:"type,omitempty"`
	Title           string      `json:"title,omitempty"`
	Description     string      `json:"description,omitempty"`
	AuthorName      string      `json:"author_name,omitempty"`
	AuthorURL       string      `json:"author_url,omitempty"`
	AuthorThumbnail string      `json:"author_thumbnail_url,omitempty"`
	AuthorID        string      `json:"author_id,omitempty"`
	AuthorLogin     string      `json:"author_login,omitempty"`
	ThumbnailURL    string      `json:"thumbnail_url,omitempty"`
	ThumbnailWidth  int         `json:"thumbnail_width,omitempty"`
	ThumbnailHeight int         `json:"thumbnail_height,omitempty"`
	Thumbnails      *Thumbnails `json:"thumbnail_urls,omitempty"`
	PlayerHTML      string      `json:"html,omitempty"`
	Width           int         `json:"width,omitempty"` // required if HTML or URL are specified
	Height          int         `json:"height,omitempty"`
	ProviderName    string      `json:"provider_name,omitempty"`
	ProviderURL     string      `json:"provider_url,omitempty"`
	URL             string      `json:"url,omitempty"`

	// Twitch VOD/Clips specific
	CreatedAt       *time.Time `json:"created_at,omitempty"`
	Game            string     `json:"game,omitempty"`
	VideoLength     int        `json:"video_length,omitempty"`
	ViewCount       int        `json:"view_count,omitempty"`
	TwitchType      string     `json:"twitch_type,omitempty"`
	TwitchURLType   string     `json:"twitch_url_type,omitempty"`
	TwitchContentID string     `json:"twitch_content_id,omitempty"`
	TwitchEntity    string     `json:"twitch_entity,omitempty"`
	StartTime       *time.Time `json:"start_time,omitempty"`
	EndTime         *time.Time `json:"end_time,omitempty"`

	// Twitch Clips specific
	CuratorName string `json:"curator_name,omitempty"`
	CuratorURL  string `json:"curator_url,omitempty"`

	// Twitch Stream specific
	StreamType string `json:"stream_type,omitempty"`

	defaultTTL time.Duration
	streamTTL  time.Duration
	eventTTL   time.Duration
	vodTTL     time.Duration
}

// CacheTTL returns the TTL for the embed type
func (e Embed) CacheTTL() time.Duration {
	switch e.TwitchType {
	case "stream":
		if e.TwitchContentID != "" {
			return e.streamTTL
		}
		// Set TTL to 1 week if the stream is offline
		return time.Hour * 24 * 7
	case "vod":
		return e.vodTTL
	case "event":
		return e.eventTTL
	default:
		return e.defaultTTL
	}
}

// Oembed  contains embed information for non-twitch content eg: YouTube, Vimeo, Reddit
//
// Note: The Oembed spec can be found at: http://oembed.com
type Oembed struct {
	Type         string        `json:"type"`
	Version      string        `json:"version"` // Must always be 1.0
	Title        string        `json:"title"`
	AuthorName   string        `json:"author_name"`
	AuthorURL    string        `json:"author_url"`
	ProviderName string        `json:"provider_name"`
	ProviderURL  string        `json:"provider_url"`
	CacheAge     time.Duration `json:"cache_age"`

	// optional fields
	HTML   string `json:"html,omitempty"`  // required if type == video|rich
	URL    string `json:"url,omitempty"`   // required if type == photo
	Width  int    `json:"width,omitempty"` // required if HTML or URL are specified
	Height int    `json:"height,omitempty"`

	// thumbnail fields
	ThumbnailURL    string `json:"thumbnail_url,omitempty"`
	ThumbnailWidth  int    `json:"thumbnail_width,omitempty"`
	ThumbnailHeight int    `json:"thumbnail_height,omitempty"`
}

// Thumbnails provides additional thumbnail URLs
type Thumbnails struct {
	Large  string `json:"large,omitempty"`
	Medium string `json:"medium,omitempty"`
	Small  string `json:"small,omitempty"`
	Tiny   string `json:"tiny,omitempty"`
}

// URLAndOembed contains the oembed associated with an URL
type URLAndOembed struct {
	URL    string  `json:"url"`
	Oembed *Oembed `json:"oembed"`
}

// OembedsForURLs contains a data returned by getEmbedsForURLs
type OembedsForURLs struct {
	Oembeds []URLAndOembed `json:"oembeds"`
}

// URLAndEntity contains the entity associated with an URL
type URLAndEntity struct {
	URL    string        `json:"url"`
	Entity entity.Entity `json:"entity"`
}

// EntitiesForURLs contains data returned by /v1/entities_for_urls
type EntitiesForURLs struct {
	Entities []URLAndEntity `json:"entities"`
}
