package provider

import (
	"fmt"
	"net/http"
	"net/url"
	"regexp"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"golang.org/x/net/context"
)

var (
	imgur               = regexp.MustCompile(`^(https?:\/\/)?(www\.|i\.)?imgur\.com\/`)
	gfycat              = regexp.MustCompile(`^(https?:\/\/)?(www\.)?gfycat\.com\/`)
	redditUpload        = regexp.MustCompile(`^(https?:\/\/)?i\.reddituploads\.com\/`)
	embedlyCDN          = regexp.MustCompile(`^(https?:\/\/)?cdn\.embedly\.com\/`)
	embedlyProviderList = []*regexp.Regexp{imgur, gfycat, redditUpload, embedlyCDN}
)

// EmbedlyConfig configures the embed request URL and API Key
type EmbedlyConfig struct {
	embedlyOembedHost *distconf.Str
	enableEmbedly     *distconf.Bool
	APIKey            *distconf.Str
}

// Load EmbedlyConfig from distconf
func (ec *EmbedlyConfig) Load(dconf *distconf.Distconf) error {
	ec.embedlyOembedHost = dconf.Str("shine.embedly_oembed_host", "http://api.embed.ly")
	ec.enableEmbedly = dconf.Bool("shine.enable_embedly", false)
	ec.APIKey = dconf.Str("shine.embedly_api_key", "")
	if ec.APIKey.Get() == "" {
		return fmt.Errorf("cannot find shine.embedly_api_key")
	}
	return nil
}

// Embedly handles embed requests
type Embedly struct {
	Config *EmbedlyConfig
	NewReq clients.NewHTTPRequest
	Client clients.RequestDoer
}

// Matches confirms the passed-in url matches the provider
func (e *Embedly) Matches(embedURL string) bool {
	if !e.Config.enableEmbedly.Get() {
		return false
	}
	for _, regexURL := range embedlyProviderList {
		if regexURL.MatchString(embedURL) {
			return true
		}
	}
	return false
}

// EntityForURL returns entity derived from embedURL
// EntityForURL returns an error if embedURL can't be converted to an entity
func (e *Embedly) EntityForURL(embedURL string) (entity.Entity, error) {
	if !e.Matches(embedURL) {
		return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
	}
	return entity.New(entity.NamespaceOembed, embedURL), nil
}

// RequestEmbed requests embed information for the passed-in url
func (e *Embedly) RequestEmbed(ctx context.Context, embedURL string, autoplay bool) (*api.Embed, error) {
	var response api.Embed
	path := "/1/oembed"
	query := e.queryParams(embedURL)
	if err := clients.DoHTTP(ctx, &http.Client{}, "GET", e.Config.embedlyOembedHost.Get()+path, query, nil, &response, e.NewReq); err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	return sanitizeEmbed(&response, false, e, embedURL, autoplay)
}

// RequestOembed requests oembed information for the passed-in url
func (e *Embedly) RequestOembed(ctx context.Context, oembedURL string, autoplay bool) (*api.Oembed, error) {
	var response api.Oembed
	path := "/1/oembed"
	query := e.queryParams(oembedURL)
	if err := clients.DoHTTP(ctx, &http.Client{}, "GET", e.Config.embedlyOembedHost.Get()+path, query, nil, &response, e.NewReq); err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	return sanitizeOembed(&response, false, e.Matches, oembedURL, autoplay)
}

func (e *Embedly) queryParams(embedURL string) url.Values {
	query := url.Values{}
	query.Add("url", embedURL)
	query.Add("key", e.Config.APIKey.Get())
	query.Add("image_width", "500")
	query.Add("image_height", "281")
	query.Add("image_method", "crop")
	query.Add("secure", "true")
	return query
}
