package provider

import (
	"strconv"
	"testing"

	"code.justin.tv/feeds/distconf"
	. "github.com/smartystreets/goconvey/convey"
)

func verifyURLs(URLList []string, shouldBe func(interface{}, ...interface{}) string, matches func(string) bool) {
	for _, url := range URLList {
		So(matches(url), shouldBe)
	}
}

func TestUnit_Utils(t *testing.T) {
	Convey("Provider Utils", t, func() {
		Convey("addQueryParams should add autoplay query param to the iframe source attribute", func() {
			source := "https://clips.twitch.tv/embed?clip=proleaguecsgo/NiceOwlSMOrc"
			autoplay := true
			autoplaySource := addQueryParams(source, autoplay)
			So(source, ShouldNotEqual, autoplaySource)
			So(autoplaySource, ShouldContainSubstring, "autoplay="+strconv.FormatBool(autoplay))

			autoplay = false
			autoplaySource = addQueryParams(source, autoplay)
			So(source, ShouldNotEqual, autoplaySource)
			So(autoplaySource, ShouldContainSubstring, "autoplay="+strconv.FormatBool(autoplay))
		})
	})

	Convey("Providers Matches", t, func() {
		Convey("Twitch Collection regex function captures correct URLs", func() {
			distConfig := distconf.Distconf{}
			twitchCollectionConfig := &TwitchCollectionConfig{
				twitchCollectionHost:   distConfig.Str("shine.host", ""),
				enableTwitchCollection: distConfig.Bool("shine.enable", true),
				krakenClientID:         distConfig.Str("kraken_client_id", ""),
			}
			twitchCollection := TwitchCollection{Config: twitchCollectionConfig}
			goodURLs := []string{
				"twitch.tv/collections/nkREppjluhQLEw",
				"http://twitch.tv/collections/nkREppjluhQLEw",
				"https://twitch.tv/collections/nkREppjluhQLEw",
				"http://www.twitch.tv/collections/nkREppjluhQLEw",
				"https://www.twitch.tv/collections/nkREppjluhQLEw",
			}
			badURLs := []string{
				"https://twitch.tv/collections",
				"https://twitch.tv/collections/",
			}
			verifyURLs(goodURLs, ShouldBeTrue, twitchCollection.Matches)
			verifyURLs(badURLs, ShouldBeFalse, twitchCollection.Matches)
		})

		Convey("Twitch Stream regex function captures correct URLs", func() {
			distConfig := distconf.Distconf{}
			twitchStreamConfig := &TwitchStreamConfig{
				twitchStreamHost:   distConfig.Str("shine.host", ""),
				enableTwitchStream: distConfig.Bool("shine.enable", true),
			}
			twitchStream := TwitchStream{Config: twitchStreamConfig}
			goodURLs := []string{
				"twitch.tv/broadcasts/4564564/channel/454564564",
				"http://twitch.tv/broadcasts/4564564/channel/454564564",
				"https://twitch.tv/broadcasts/4564564/channel/454564564",
			}
			badURLs := []string{
				"https://twitch.tv/broadcasts/45fdf64564/channel/454564564",
				"https://twitch.tv/broadcasts/4564564/channel/4fdf54564564",
				"https://twitch.tv/broadcast/4564564/channel/4fdf54564564",
				"https://twitch.tv/broadcasts/4564564/channels/4fdf54564564",
			}
			verifyURLs(goodURLs, ShouldBeTrue, twitchStream.Matches)
			verifyURLs(badURLs, ShouldBeFalse, twitchStream.Matches)
		})

		Convey("Twitch VOD regex function captures correct URLs", func() {
			distConfig := distconf.Distconf{}
			twitchVODConfig := &TwitchVODConfig{
				twitchVODHost:   distConfig.Str("shine.host", ""),
				enableTwitchVOD: distConfig.Bool("shine.enable", true),
			}
			twitchVOD := TwitchVOD{Config: twitchVODConfig}
			goodURLs := []string{
				"https://www.twitch.tv/lirik/v/117022022",
				"http://www.twitch.tv/lirik/v/117022022",
				"https://twitch.tv/lirik/v/117022022",
				"www.twitch.tv/lirik/v/117022022",
				"twitch.tv/lirik/v/117022022",
				"https://www.twitch.tv/videos/117022022",
				"http://www.twitch.tv/videos/117022022",
				"https://twitch.tv/videos/117022022",
				"www.twitch.tv/videos/117022022",
				"twitch.tv/videos/117022022",
				"https://player.twitch.tv/?video=v135535642",
				"http://player.twitch.tv/?video=v135535642",
			}
			badURLs := []string{
				"https://www.twitch.tv/lirik/c/117022022",
				"https://www.twitch.tv/lirik/clips/v/117022022",
				"https//www.twitch.tv/lirik/v/117022022",
				"https://ww.twitch.tv/lirik/v/117022022",
				"https://www.twitch.tv/video/117022022",
				"https://www.twitch.tv/video/117022022kjklj",
				"https://www.twitch.tv/video",
				"https://www.twitchtv//videos/117022022",
				"https://www.player.twitch.tv/?video=v135535642",
				"https://player.twitch.tv/?v=135535642",
			}
			verifyURLs(goodURLs, ShouldBeTrue, twitchVOD.Matches)
			verifyURLs(badURLs, ShouldBeFalse, twitchVOD.Matches)
		})

		Convey("Twitch Clip regex function captures correct URLs", func() {
			distConfig := distconf.Distconf{}
			twitchClipConfig := &TwitchClipConfig{
				twitchClipHost:   distConfig.Str("shine.host", ""),
				enableTwitchClip: distConfig.Bool("shine.enable", true),
			}
			twitchClip := TwitchClip{Config: twitchClipConfig}
			goodURLs := []string{
				"https://clips.twitch.tv/BusyNewtNotLikeThis",
				"https://clips.twitch.tv/BusyNewtNotLikeThis123",
				"https://clips.twitch.tv/BusyNewtNotLikeThis",
				"https://clips.twitch.tv/BusyNe233wtNotLikeThis",
				"https://clips.twitch.tv/BusyNe233wtNotLike--This",
				"https://clips.twitch.tv/jfkd5423lsajkl",
				"http://clips.twitch.tv/BusyNewtNotLikeThis",
				"clips.twitch.tv/BusyNewtNotLikeThis",
				"http://www.twitch.tv/weirdchannelname/clip/BusyNewtNotLikeThis",
				"http://www.twitch.tv/weirdchannelname/clip/BusyNewtNotLikeThis123",
				"http://www.twitch.tv/weirdchannelname/clip/BusyNe233wtNotLikeThis",
				"http://www.twitch.tv/weirdchannelname/clip/BusyNe233wtNotLike--This",
				"http://www.twitch.tv/weirdchannelname/clip/jfkd5423lsajkl",
				"twitch.tv/weirdchannelname/clip/jfkd5423lsajkl",
			}
			badURLs := []string{
				"https://clipstwitch.tv/odpixel/BusyNewtNotLikeThis",
				"https://clips.twitchtv/odpixel/BusyNewtNotLikeThis",
				"https://clips.twitch.tv/@odpixel/BusyNewtNotLikeThis",
				"https://clips.twitch.tv/@odpixel/BusyNewtNotLikeThis@",
				"https://clips.twitch.tv/test123/BusyNewtNotLikeThis",
				"http://wwwtwitch.tv/weirdchannelname/clip/jfkd5423lsajkl",
				"http://www.twitch.tv/@weirdchannelname/clip/jfkd5423lsajkl",
				"http://www.twitch.tv/@weirdchannelname/clip/jfkd5423lsajkl@",
				"http://www.twitch.tv/@weirdchannelname/test123/clip/jfkd5423lsajkl@",
			}
			verifyURLs(goodURLs, ShouldBeTrue, twitchClip.Matches)
			verifyURLs(badURLs, ShouldBeFalse, twitchClip.Matches)
		})

		Convey("Twitch Event regex function captures correct URLs", func() {
			distConfig := distconf.Distconf{}
			twitchEventConfig := &TwitchEventConfig{
				twitchEventHost:   distConfig.Str("shine.host", ""),
				enableTwitchEvent: distConfig.Bool("shine.enable", true),
				krakenClientID:    distConfig.Str("kraken_client_id", ""),
			}
			twitchEvent := TwitchEvent{Config: twitchEventConfig}
			goodURLs := []string{
				"https://www.twitch.tv/events/45684",
				"http://www.twitch.tv/events/45684",
				"https://twitch.tv/events/45684",
				"http://twitch.tv/events/45684",
				"twitch.tv/events/45684",
				"https://www.twitch.tv/events/45684",
				"https://www.twitch.tv/events/-2hCDW_SSIazplQcTbYLnw",
			}
			badURLs := []string{
				"htt://www.twitch.tv/events/45684",
				"http://www.twitch.tv/event/45684",
				"https://stwitch.tv/events/45684",
				"http://twitch.tvs/events/45684",
				"twitch.tv/event/45684",
				"https://www.twitch.tv/events/45684/jdjd",
			}
			verifyURLs(goodURLs, ShouldBeTrue, twitchEvent.Matches)
			verifyURLs(badURLs, ShouldBeFalse, twitchEvent.Matches)
		})

		Convey("Youtube regex function captures correct URLs", func() {
			distConfig := distconf.Distconf{}
			youtubeConfig := &YoutubeConfig{
				youtubeOembedHost: distConfig.Str("shine.host", ""),
				enableYoutube:     distConfig.Bool("shine.enable", true),
			}
			youtube := Youtube{Config: youtubeConfig}
			goodURLs := []string{
				"https://www.youtube.com/watch?v=xsSnOQynTHs",
				"http://www.youtube.com/watch?v=xsSnOQynTHs",
				"www.youtube.com/watch?v=xsSnOQynTHs",
				"youtube.com/watch?v=xsSnOQynTHs",
				"https://youtube.com/watch?v=xsSnOQynTHs",
				"https://youtu.be/ZNPn6aqildk",
				"http://youtu.be/ZNPn6aqildk",
				"youtu.be/ZNPn6aqildk",
			}
			badURLs := []string{
				"htt://www.youtube.com/watch?v=xsSnOQynTHs",
				"https//www.youtube.com/watch?v=xsSnOQynTHs",
				"https://outube.com/watch?v=xsSnOQynTHs",
				"https://www.youtubecom/watch?v=xsSnOQynTHs",
				"https://wwwyoutube.com/watch?v=xsSnOQynTHs",
				"https://www.youtu.be/ZNPn6aqildk",
				"https://youtu.be.com/ZNPn6aqildk",
			}
			verifyURLs(goodURLs, ShouldBeTrue, youtube.Matches)
			verifyURLs(badURLs, ShouldBeFalse, youtube.Matches)
		})

		Convey("Embedly regex function captures correct URLs", func() {
			distConfig := distconf.Distconf{}
			embedlyConfig := &EmbedlyConfig{
				embedlyOembedHost: distConfig.Str("shine.host", ""),
				enableEmbedly:     distConfig.Bool("shine.enable", true),
			}
			embedly := Embedly{Config: embedlyConfig}
			goodURLs := []string{
				"https://www.gfycat.com/VainAlarmedGreathornedowl",
				"http://www.gfycat.com/VainAlarmedGreathornedowl",
				"www.gfycat.com/VainAlarmedGreathornedowl",
				"gfycat.com/VainAlarmedGreathornedowl",
				"https://gfycat.com/VainAlarmedGreathornedowl",
				"https://www.imgur.com/evmMnEH",
				"http://www.imgur.com/evmMnEH",
				"https://i.imgur.com/evmMnEH",
				"imgur.com/evmMnEH",
				"http://imgur.com/evmMnEH",
				"https://i.reddituploads.com/f9c5531442e545bba57dee098a11747b",
				"http://i.reddituploads.com/f9c5531442e545bba57dee098a11747b",
				"i.reddituploads.com/f9c5531442e545bba57dee098a11747b",
				"http://cdn.embedly.com/",
				"https://cdn.embedly.com/",
				"cdn.embedly.com/",
			}
			badURLs := []string{
				"htt://www.gfycat.com/VainAlarmedGreathornedowl",
				"https//www.gfycat.com/VainAlarmedGreathornedowl",
				"https://wwwfycat.com/VainAlarmedGreathornedowl",
				"https://www.gfycatcom/VainAlarmedGreathornedowl",
				"https://wwwgfycat.com/VainAlarmedGreathornedowl",
				"htt://www.imgur.com/evmMnEH",
				"https//www.imgur.com/evmMnEH",
				"https://wwwimgur.com/evmMnEH",
				"https://www.imgurcom/evmMnEH",
				"https://k.imgur.com/evmMnEH",
				"htt://i.reddituploads.com/f9c5531442e545bba57dee098a11747b",
				"https//i.reddituploads.com/f9c5531442e545bba57dee098a11747b",
				"https://ireddituploads.com/f9c5531442e545bba57dee098a11747b",
				"https://i.reddituploadscom/f9c5531442e545bba57dee098a11747b",
				"https://www.reddituploads.com/f9c5531442e545bba57dee098a11747b",
				"htt://cdn.embedly.com/",
				"https//cdn.embedly.com/",
				"cdn.embedly.com",
			}
			for _, url := range goodURLs {
				So(embedly.Matches(url), ShouldBeTrue)
			}
			for _, url := range badURLs {
				So(embedly.Matches(url), ShouldBeFalse)
			}
			verifyURLs(goodURLs, ShouldBeTrue, embedly.Matches)
			verifyURLs(badURLs, ShouldBeFalse, embedly.Matches)
		})

		Convey("Vimeo regex function captures correct URLs", func() {
			distConfig := distconf.Distconf{}
			vimeoConfig := &VimeoConfig{
				vimeoOembedHost: distConfig.Str("shine.host", ""),
				enableVimeo:     distConfig.Bool("shine.enable", true),
			}
			vimeo := Vimeo{Config: vimeoConfig}
			goodURLs := []string{
				"https://www.vimeo.com/188568367",
				"http://www.vimeo.com/188568367",
				"https://vimeo.com/188568367",
				"vimeo.com/188568367",
				"https://player.vimeo.com/188568367",
				"http://player.vimeo.com/188568367",
				"player.vimeo.com/188568367",
			}
			badURLs := []string{
				"htt://www.vimeo.com/188568367",
				"https//www.vimeo.com/188568367",
				"https://www.vimeocom/188568367",
				"https://wwwvimeo.com/188568367",
				"htt://player.vimeo.com/188568367",
				"https//player.vimeo.com/188568367",
				"https://player.vimeocom/188568367",
				"https://playervimeo.com/188568367",
			}
			verifyURLs(goodURLs, ShouldBeTrue, vimeo.Matches)
			verifyURLs(badURLs, ShouldBeFalse, vimeo.Matches)
		})
	})
}
