package provider

import (
	"fmt"
	"net/http"
	"regexp"
	"strconv"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"golang.org/x/net/context"
)

const (
	clipHTMLWidth  = 620
	clipHTMLHeight = 351
)

var (
	twitchClipRegex        = regexp.MustCompile(`^(?:https?://)?clips\.twitch\.tv/(?:embed\?(?:clip=|autoplay=))?([a-zA-Z0-9_&=-]+)$`)
	twitchClipOnsiteRegex  = regexp.MustCompile(`^(?:https?://)?(?:www?\.)?twitch\.tv((/\w+)/)clip/([a-zA-Z0-9_?&=-]+)$`)
	twitchClipProviderList = []*regexp.Regexp{twitchClipRegex, twitchClipOnsiteRegex}
	clipHTMLIframe         = `<iframe src="https://clips.twitch.tv/embed?clip=%s&autoplay=%s" width="620" height="351" frameborder="0" scrolling="no" allowfullscreen></iframe>`
)

// clipResponse is the response struct for GET clip
type clipResponse struct {
	TwitchContentID string         `json:"cid"`
	Slug            string         `json:"id"`
	Title           string         `json:"title"`
	AuthorName      string         `json:"broadcaster_display_name"`
	AuthorURL       string         `json:"broadcaster_channel_url"`
	AuthorThumbnail string         `json:"broadcaster_logo"`
	AuthorID        string         `json:"broadcaster_id"`
	AuthorLogin     string         `json:"broadcaster_login"`
	CuratorName     string         `json:"curator_display_name"`
	CuratorURL      string         `json:"curator_channel_url"`
	Thumbnails      api.Thumbnails `json:"thumbnails"`
	CreatedAt       *time.Time     `json:"created_at"`
	Game            string         `json:"game"`
	VideoLength     float64        `json:"duration"`
	ViewCount       int64          `json:"views"`
	ThumbnailURL    string         `json:"preview_image"`
}

// TwitchClipConfig configures the embed request URL
type TwitchClipConfig struct {
	enableTwitchClip *distconf.Bool
	twitchClipHost   *distconf.Str
}

// Load TwitchClipConfig from distconf
func (tc *TwitchClipConfig) Load(dconf *distconf.Distconf) error {
	tc.enableTwitchClip = dconf.Bool("shine.enable_twitch_clips", false)
	tc.twitchClipHost = dconf.Str("shine.twitch_clip_host", "")
	return nil
}

// TwitchClip handles embed requests
type TwitchClip struct {
	Config *TwitchClipConfig
	NewReq clients.NewHTTPRequest
	Client clients.RequestDoer
}

// Matches confirms the passed-in url matches the provider
func (t *TwitchClip) Matches(embedURL string) bool {
	if !t.Config.enableTwitchClip.Get() {
		return false
	}
	for _, regexURL := range twitchClipProviderList {
		if regexURL.MatchString(embedURL) {
			return true
		}
	}
	return false
}

// EntityForURL returns entity derived from embedURL
// EntityForURL returns an error if embedURL can't be converted to an entity
func (t *TwitchClip) EntityForURL(embedURL string) (entity.Entity, error) {
	if !t.Matches(embedURL) {
		return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
	}

	if slug := t.getClipSlug(embedURL); slug != "" {
		return entity.New(entity.NamespaceClip, slug), nil
	}
	return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
}

// getClipSlug returns the clip slug from the embed URL
func (t *TwitchClip) getClipSlug(embedURL string) string {
	matches := twitchClipRegex.FindStringSubmatch(embedURL)
	otherMatches := twitchClipOnsiteRegex.FindStringSubmatch(embedURL)

	if len(matches) > 1 {
		return matches[1]
	}

	if len(otherMatches) > 1 {
		return otherMatches[3]
	}

	return ""
}

func (t *TwitchClip) getClipResponse(ctx context.Context, embedURL string) (*clipResponse, error) {
	clip := &clipResponse{}
	slug := t.getClipSlug(embedURL)
	path := "/api/v1/clips/" + slug
	err := clients.DoHTTP(ctx, t.Client, "GET", t.Config.twitchClipHost.Get()+path, nil, nil, clip, nil)
	return clip, err
}

// RequestEmbed requests embed information for the passed-in url
func (t *TwitchClip) RequestEmbed(ctx context.Context, embedURL string, autoplay bool) (*api.Embed, error) {
	clip, err := t.getClipResponse(ctx, embedURL)
	if err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	embed := convertClipToEmbed(clip, embedURL, autoplay)
	return sanitizeEmbed(embed, true, t, embedURL, autoplay)
}

// RequestOembed requests oembed information for the passed-in url
func (t *TwitchClip) RequestOembed(ctx context.Context, embedURL string, autoplay bool) (*api.Oembed, error) {
	clip, err := t.getClipResponse(ctx, embedURL)
	if err != nil {
		switch errorCode(err) {
		case http.StatusNotFound, http.StatusForbidden:
			return nil, nil
		}
		return nil, err
	}
	embed := convertClipToOembed(clip, embedURL, autoplay)
	return sanitizeOembed(embed, true, t.Matches, embedURL, autoplay)
}

func convertClipToEmbed(clip *clipResponse, embedURL string, autoplay bool) *api.Embed {
	playerHTML := fmt.Sprintf(clipHTMLIframe, clip.Slug, strconv.FormatBool(autoplay))
	return &api.Embed{
		Type:            "video",
		TwitchType:      entity.NamespaceClip,
		ProviderName:    "Twitch",
		RequestURL:      embedURL,
		PlayerHTML:      playerHTML,
		Width:           clipHTMLWidth,
		Height:          clipHTMLHeight,
		Title:           clip.Title,
		AuthorName:      clip.AuthorName,
		AuthorURL:       clip.AuthorURL,
		AuthorThumbnail: clip.AuthorThumbnail,
		AuthorID:        clip.AuthorID,
		AuthorLogin:     clip.AuthorLogin,
		ThumbnailURL:    clip.ThumbnailURL,
		Thumbnails: &api.Thumbnails{
			Medium: clip.Thumbnails.Medium,
			Small:  clip.Thumbnails.Small,
			Tiny:   clip.Thumbnails.Tiny,
		},
		CreatedAt:       clip.CreatedAt,
		Game:            clip.Game,
		VideoLength:     int(clip.VideoLength),
		ViewCount:       int(clip.ViewCount),
		TwitchContentID: clip.TwitchContentID,
		CuratorName:     clip.CuratorName,
		CuratorURL:      clip.CuratorURL,
	}
}

func convertClipToOembed(clip *clipResponse, embedURL string, autoplay bool) *api.Oembed {
	playerHTML := fmt.Sprintf(clipHTMLIframe, clip.Slug, strconv.FormatBool(autoplay))
	oembed := &api.Oembed{
		Type:         "video",
		Version:      "1.0",
		Title:        clip.Title,
		AuthorName:   clip.AuthorName,
		AuthorURL:    clip.AuthorURL,
		ProviderName: "Twitch",

		HTML:   playerHTML,
		Width:  clipHTMLWidth,
		Height: clipHTMLHeight,
	}
	return oembed
}
