package provider

import (
	"fmt"
	"io"
	"net/http"
	"regexp"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"golang.org/x/net/context"
)

var (
	twitchCollectionRegex = regexp.MustCompile(`^(https?:\/\/)?(www\.)?twitch\.tv\/collections\/([a-zA-Z0-9]+)$`)
	twitchProviderList    = []*regexp.Regexp{twitchCollectionRegex}
)

const (
	collectionHTMLHeight = 281
	collectionHTMLWidth  = 500
	collectionQuery      = `
query getCollection($id: ID!) {
  collection(id: $id) {
    title,
    viewCount,
    thumbnailURL,
    owner {
      displayName,
      profileImageURL(width:600),
      id,
      login,
    }
    items{
      edges {
        node {
          __typename
          ... on Video {
            id,
            createdAt,
            game {
              name,
            }
            lengthSeconds,
            description,
            thumbnailURLs
          }
        }
      }
    }
  }
}`
)

type twitchCollectionQueryVariables struct {
	ID string `json:"id"`
}

type twitchCollectionQuery struct {
	Query     string                         `json:"query"`
	Variables twitchCollectionQueryVariables `json:"variables"`
}

type twitchCollectionResponse struct {
	Data struct {
		Collection struct {
			Title        string `json:"title"`
			ViewCount    int    `json:"viewCount"`
			ThumbnailURL string `json:"thumbnailURL"`
			Owner        struct {
				DisplayName     string `json:"displayName"`
				ProfileImageURL string `json:"profileImageURL"`
				ID              string `json:"id"`
				Login           string `json:"login"`
			} `json:"owner"`
			Items struct {
				Edge []struct {
					Node struct {
						Typename  string     `json:"__typename"`
						ID        string     `json:"id"`
						CreatedAt *time.Time `json:"createdAt"`
						Game      struct {
							Name string `json:"name"`
						} `json:"game"`
						LengthSeconds int      `json:"lengthSeconds"`
						Description   string   `json:"description"`
						ThumbnailURLs []string `json:"thumbnailURLs"`
					} `json:"node"`
				} `json:"edges"`
			} `json:"items"`
		} `json:"collection"`
	} `json:"data"`
}

// TwitchCollectionConfig configures the embed request URL
type TwitchCollectionConfig struct {
	twitchCollectionHost   *distconf.Str
	enableTwitchCollection *distconf.Bool
	krakenClientID         *distconf.Str
}

// Load TwitchCollectionConfig from distconf
func (tc *TwitchCollectionConfig) Load(dconf *distconf.Distconf) error {
	tc.twitchCollectionHost = dconf.Str("shine.twitch_collection_host_new", "https://api.twitch.tv")
	tc.enableTwitchCollection = dconf.Bool("shine.enable_twitch_collection", false)
	tc.krakenClientID = dconf.Str("kraken_client_id", "")
	return nil
}

// TwitchCollection handles embed requests
type TwitchCollection struct {
	Config *TwitchCollectionConfig
	NewReq clients.NewHTTPRequest
	Client clients.RequestDoer
}

// Matches confirms the passed-in url matches the provider
func (t *TwitchCollection) Matches(embedURL string) bool {
	if !t.Config.enableTwitchCollection.Get() {
		return false
	}
	for _, regexURL := range twitchProviderList {
		if regexURL.MatchString(embedURL) {
			return true
		}
	}
	return false
}

func (t *TwitchCollection) twitchAPINewHTTPRequest(ctx context.Context, method string, url string, body io.Reader) (*http.Request, error) {
	req, err := http.NewRequest(method, url, body)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Client-ID", t.Config.krakenClientID.Get())
	req = req.WithContext(ctx)
	return req, nil
}

// EntityForURL returns entity derived from embedURL
// EntityForURL returns an error if embedURL can't be converted to an entity
func (t *TwitchCollection) EntityForURL(embedURL string) (entity.Entity, error) {
	if !t.Matches(embedURL) {
		return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
	}
	return entity.New(entity.NamespaceOembed, embedURL), nil
}

func getCollectionIDFromURL(url string) string {
	matches := twitchCollectionRegex.FindStringSubmatch(url)
	if len(matches) > 0 {
		return matches[len(matches)-1]
	}
	return ""
}

func getThumbnail(url string, width string, height string) string {
	return strings.Replace(strings.Replace(url, "{width}", width, 1), "{height}", height, 1)
}

// RequestEmbed requests embed information for the passed-in url
func (t *TwitchCollection) RequestEmbed(ctx context.Context, embedURL string, autoplay bool) (*api.Embed, error) {
	path := "/gql"
	collectionID := getCollectionIDFromURL(embedURL)
	body := twitchCollectionQuery{
		Query: collectionQuery,
		Variables: twitchCollectionQueryVariables{
			ID: collectionID,
		},
	}
	var response twitchCollectionResponse
	if err := clients.DoHTTP(ctx, t.Client, "POST", t.Config.twitchCollectionHost.Get()+path, nil, body, &response, t.twitchAPINewHTTPRequest); err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}

	result := api.Embed{
		RequestURL:      embedURL,
		Type:            "video",
		Title:           response.Data.Collection.Title,
		AuthorName:      response.Data.Collection.Owner.DisplayName,
		AuthorURL:       "https://www.twitch.tv/" + response.Data.Collection.Owner.Login + "/profile",
		AuthorThumbnail: response.Data.Collection.Owner.ProfileImageURL,
		AuthorID:        response.Data.Collection.Owner.ID,
		AuthorLogin:     response.Data.Collection.Owner.Login,
		ProviderName:    "Twitch",
		URL:             embedURL,
		ViewCount:       response.Data.Collection.ViewCount,
		TwitchType:      entity.NamespaceVod,
	}

	defaultVodIndex := 0
	for defaultVodIndex < len(response.Data.Collection.Items.Edge) && response.Data.Collection.Items.Edge[defaultVodIndex].Node.Typename != "Video" {
		defaultVodIndex++
	}
	if defaultVodIndex < len(response.Data.Collection.Items.Edge) {
		vod := response.Data.Collection.Items.Edge[defaultVodIndex].Node
		result.Description = vod.Description

		if len(vod.ThumbnailURLs) > 0 {
			thumbnailURL := vod.ThumbnailURLs[0]
			result.Thumbnails = &api.Thumbnails{
				Large:  getThumbnail(thumbnailURL, "1920", "1080"),
				Medium: getThumbnail(thumbnailURL, "1280", "720"),
				Small:  getThumbnail(thumbnailURL, "640", "360"),
				Tiny:   getThumbnail(thumbnailURL, "128", "72"),
			}
			result.ThumbnailURL = result.Thumbnails.Small
		}

		result.PlayerHTML = fmt.Sprintf(iframeVODHTML, "v"+vod.ID+"&collection="+collectionID, strconv.FormatBool(autoplay))
		result.Width = collectionHTMLWidth
		result.Height = collectionHTMLHeight
		result.CreatedAt = vod.CreatedAt
		result.Game = vod.Game.Name
		result.VideoLength = vod.LengthSeconds
	}
	return sanitizeEmbed(&result, true, t, embedURL, autoplay)
}

// RequestOembed requests oembed information for the passed-in url
func (t *TwitchCollection) RequestOembed(ctx context.Context, embedURL string, autoplay bool) (*api.Oembed, error) {
	embed, err := t.RequestEmbed(ctx, embedURL, autoplay)
	if err != nil {
		return nil, err
	}

	if embed == nil {
		return nil, nil
	}
	oembed := convertCollectionEmbedtoOembed(embed)
	return sanitizeOembed(oembed, true, t.Matches, embedURL, autoplay)
}

func convertCollectionEmbedtoOembed(embed *api.Embed) *api.Oembed {
	return &api.Oembed{
		Type:         "video",
		Version:      "1.0",
		Title:        embed.Title,
		ProviderName: embed.ProviderName,
		AuthorName:   embed.AuthorName,
		AuthorURL:    embed.AuthorURL,

		HTML:   embed.PlayerHTML,
		Width:  collectionHTMLWidth,
		Height: collectionHTMLHeight,
	}
}
