package provider

import (
	"io"
	"net/http"
	"regexp"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"golang.org/x/net/context"
)

const (
	embedThumbnailWidth  = "500"
	embedThumbnailHeight = "281"

	embedThumbnailWidthInt  = 500
	embedThumbnailHeightInt = 281
)

// EventResponse is the top level event data
type EventResponse struct {
	EventImage  string     `json:"cover_image_url"`
	Title       string     `json:"title"`
	Description string     `json:"description"`
	Channel     Channel    `json:"channel"`
	Game        Game       `json:"game"`
	StartTime   *time.Time `json:"start_time"`
	EndTime     *time.Time `json:"end_time"`
	ID          string     `json:"_id"`
}

// Channel is the object referenced in EventResponse
type Channel struct {
	DisplayName     string `json:"display_name"`
	AuthorID        int    `json:"_id"`
	AuthorURL       string `json:"url"`
	AuthorThumbnail string `json:"logo"`
	AuthorLogin     string `json:"name"`
}

// Game is the object referenced in EventResponse
type Game struct {
	Name string `json:"name"`
}

var twitchEventReg = regexp.MustCompile(`^(?:https?:\/\/)?(?:www\.)?twitch\.tv\/events\/([a-zA-Z0-9_=-]+)$`)

// TwitchEventConfig controls whether the provider is turned on or off
type TwitchEventConfig struct {
	twitchEventHost   *distconf.Str
	enableTwitchEvent *distconf.Bool
	krakenClientID    *distconf.Str
}

// Load TwitchEventConfig from distconf
func (te *TwitchEventConfig) Load(dconf *distconf.Distconf) error {
	te.twitchEventHost = dconf.Str("shine.twitch_event_host", "https://api.twitch.tv")
	te.enableTwitchEvent = dconf.Bool("shine.enable_twitch_event", false)
	te.krakenClientID = dconf.Str("kraken_client_id", "")
	return nil
}

// TwitchEvent handles embed requests
type TwitchEvent struct {
	Config *TwitchEventConfig
	NewReq clients.NewHTTPRequest
	Client clients.RequestDoer
}

// Matches confirms the passed-in url matches the provider
func (t *TwitchEvent) Matches(embedURL string) bool {
	if !t.Config.enableTwitchEvent.Get() {
		return false
	}
	if twitchEventReg.MatchString(embedURL) {
		return true
	}
	return false
}

// EntityForURL returns entity derived from embedURL
// EntityForURL returns an error if embedURL can't be converted to an entity
func (t *TwitchEvent) EntityForURL(embedURL string) (entity.Entity, error) {
	if !t.Matches(embedURL) {
		return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
	}
	return entity.New(entity.NamespaceEvent, getEventID(embedURL)), nil
}

func (t *TwitchEvent) getEventResponse(ctx context.Context, embedURL string) (*EventResponse, error) {
	var resp EventResponse
	eventID := getEventID(embedURL)
	path := "/v5/events/" + eventID
	url := t.Config.twitchEventHost.Get() + path
	err := clients.DoHTTP(ctx, t.Client, "GET", url, nil, nil, &resp, t.twitchAPINewHTTPRequest)
	return &resp, err
}

// RequestEmbed requests embed information for the passed-in url
func (t *TwitchEvent) RequestEmbed(ctx context.Context, embedURL string, autoplay bool) (*api.Embed, error) {
	event, err := t.getEventResponse(ctx, embedURL)
	if err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	embed := convertEventToEmbed(event, embedURL)
	return sanitizeEmbed(embed, false, t, embedURL, autoplay)
}

// RequestOembed requests oembed information for the passed-in url
func (t *TwitchEvent) RequestOembed(ctx context.Context, embedURL string, autoplay bool) (*api.Oembed, error) {
	event, err := t.getEventResponse(ctx, embedURL)
	if err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	oembed := convertEventToOembed(event, embedURL)
	return sanitizeOembed(oembed, false, t.Matches, embedURL, autoplay)
}

func (t *TwitchEvent) twitchAPINewHTTPRequest(ctx context.Context, method string, url string, body io.Reader) (*http.Request, error) {
	req, err := http.NewRequest(method, url, body)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Client-ID", t.Config.krakenClientID.Get())
	req = req.WithContext(ctx)
	return req, nil
}

func getEventID(embedURL string) string {
	matches := twitchEventReg.FindStringSubmatch(embedURL)
	if len(matches) == 2 {
		return matches[1]
	}
	return ""
}

func convertEventToEmbed(event *EventResponse, embedURL string) *api.Embed {
	thumbnailURL := strings.Replace(event.EventImage, "{width}", embedThumbnailWidth, 1)
	thumbnailURL = strings.Replace(thumbnailURL, "{height}", embedThumbnailHeight, 1)
	authorID := strconv.Itoa(event.Channel.AuthorID)
	if authorID == "0" {
		authorID = ""
	}
	embed := &api.Embed{
		RequestURL:      embedURL,
		Type:            "event",
		TwitchType:      entity.NamespaceEvent,
		ProviderName:    "Twitch",
		Title:           event.Title,
		Description:     event.Description,
		ThumbnailURL:    thumbnailURL,
		ThumbnailWidth:  embedThumbnailWidthInt,
		ThumbnailHeight: embedThumbnailHeightInt,
		Game:            event.Game.Name,
		AuthorName:      event.Channel.DisplayName,
		AuthorID:        authorID,
		AuthorLogin:     event.Channel.AuthorLogin,
		AuthorThumbnail: event.Channel.AuthorThumbnail,
		TwitchContentID: event.ID,
		StartTime:       event.StartTime,
		EndTime:         event.EndTime,
	}
	return embed
}

func convertEventToOembed(event *EventResponse, embedURL string) *api.Oembed {
	thumbnailURL := strings.Replace(event.EventImage, "{width}", embedThumbnailWidth, 1)
	thumbnailURL = strings.Replace(thumbnailURL, "{height}", embedThumbnailHeight, 1)
	oembed := &api.Oembed{
		Type:         "link",
		Version:      "1.0",
		ProviderName: "Twitch",
		Title:        event.Title,
		AuthorName:   event.Channel.DisplayName,

		ThumbnailURL:    thumbnailURL,
		ThumbnailWidth:  embedThumbnailWidthInt,
		ThumbnailHeight: embedThumbnailHeightInt,
	}
	return oembed
}
