package provider

import (
	"fmt"
	"regexp"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"golang.org/x/net/context"
)

const (
	LTVHTMLWidth          = 500
	LTVHTMLHeight         = 281
	LTVThumbnailMinWidth  = 640
	LTVThumbnailMinHeight = 360
)

var (
	twitchLTVRegex        = regexp.MustCompile(`^(?:https?://)?(?:www\.)?twitch\.tv/streams/([0-9]+)/channel/([0-9]+)$`)
	twitchLTVProviderList = []*regexp.Regexp{twitchLTVRegex}
	iframeLTVHTML         = `<iframe src="https://player.twitch.tv/?stream=%s&channelId=%s&!branding&autoplay=%s" width="500" height="281" frameborder="0" scrolling="no" allowfullscreen></iframe>`
)

// TwitchLTVConfig configures the embed request URL
type TwitchLTVConfig struct {
	enableTwitchLTV *distconf.Bool
}

// Load TwitchLTVConfig from distconf
func (tc *TwitchLTVConfig) Load(dconf *distconf.Distconf) error {
	tc.enableTwitchLTV = dconf.Bool("shine.enable_twitch_ltv", false)
	return nil
}

// TwitchLTV handles embed requests
type TwitchLTV struct {
	Config *TwitchLTVConfig
}

// Matches confirms the passed-in url matches the provider
func (t *TwitchLTV) Matches(embedURL string) bool {
	if !t.Config.enableTwitchLTV.Get() {
		return false
	}
	for _, regexURL := range twitchLTVProviderList {
		if regexURL.MatchString(embedURL) {
			return true
		}
	}
	return false
}

func getStreamAndChannelID(url string) (string, string) {
	matches := twitchLTVRegex.FindStringSubmatch(url)
	if len(matches) >= 2 {
		return matches[len(matches)-2], matches[len(matches)-1]
	}
	return "", ""
}

// RequestEmbed requests embed information for the passed-in url
func (t *TwitchLTV) RequestEmbed(ctx context.Context, embedURL string, autoplay bool) (*api.Embed, error) {
	streamID, channelID := getStreamAndChannelID(embedURL)

	/*
		This is a temporary solution. We only fill fields required by visage's APIv5's getOembed.
		It came from video team's getOembed. Type is used as video/Oembed's "TwitchURLType" here.
		TODO: Rewrite this method and fill as many fields as possible here.
	*/
	embed := &api.Embed{
		Type:            "video",
		TwitchURLType:   "ltv",
		ProviderName:    "Twitch",
		VideoLength:     30,
		RequestURL:      embedURL,
		AuthorID:        channelID,
		TwitchContentID: streamID,
		PlayerHTML:      fmt.Sprintf(iframeLTVHTML, streamID, channelID, autoplay),
		Width:           LTVHTMLWidth,
		Height:          LTVHTMLHeight,
		ProviderURL:     "https://www.twitch.tv/",
		ThumbnailWidth:  LTVThumbnailMinWidth,
		ThumbnailHeight: LTVThumbnailMinHeight,
	}

	return sanitizeEmbed(embed, true, t, embedURL, autoplay)
}

// RequestOembed requests oembed information for the passed-in url
func (t *TwitchLTV) RequestOembed(ctx context.Context, embedURL string, autoplay bool) (*api.Oembed, error) {
	streamID, channelID := getStreamAndChannelID(embedURL)

	embed := &api.Oembed{
		Version:         "1.0",
		Type:            "video",
		ProviderName:    "Twitch",
		HTML:            fmt.Sprintf(iframeLTVHTML, streamID, channelID, autoplay),
		Width:           LTVHTMLWidth,
		Height:          LTVHTMLHeight,
		ProviderURL:     "https://www.twitch.tv/",
		ThumbnailWidth:  LTVThumbnailMinWidth,
		ThumbnailHeight: LTVThumbnailMinHeight,
	}

	return sanitizeOembed(embed, true, t.Matches, embedURL, autoplay)
}

func (t *TwitchLTV) EntityForURL(url string) (entity.Entity, error) {
	if !t.Matches(url) {
		return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
	}
	return entity.New(entity.NamespaceOembed, url), nil
}
