package provider

import (
	"fmt"
	"io"
	"net/http"
	"regexp"
	"strconv"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"golang.org/x/net/context"
)

var (
	twitchStreamRegex        = regexp.MustCompile(`^(?:https?:\/\/)?(?:www\.)?twitch\.tv\/broadcasts\/[0-9]+\/channel\/([0-9]+)$`)
	twitchStreamProviderList = []*regexp.Regexp{twitchStreamRegex}
	streamHTMLIframe         = `<iframe src="https://player.twitch.tv/?channel=%s" frameborder="0" allowfullscreen="true" scrolling="no" height="281" width=“500”></iframe>`
)

const (
	streamHTMLWidth  = 500
	streamHTMLHeight = 281
)

// StreamResponse is the response struct for GET streams
type StreamResponse struct {
	Stream `json:"stream"`
}

// Stream is the stream struct for GET streams
type Stream struct {
	BroadcastID int            `json:"_id"`
	Game        string         `json:"game"`
	Viewers     int            `json:"viewers"`
	CreatedAt   *time.Time     `json:"created_at"`
	Thumbnails  api.Thumbnails `json:"preview"`
	Channel     StreamChannel  `json:"channel"`
	StreamType  string         `json:"stream_type"`
}

// StreamChannel is the channel struct for GET streams
type StreamChannel struct {
	Status          string `json:"status"`
	DisplayName     string `json:"display_name"`
	ChannelID       int    `json:"_id"`
	AuthorURL       string `json:"url"`
	AuthorThumbnail string `json:"logo"`
	AuthorLogin     string `json:"name"`
}

// TwitchStreamConfig configures the embed request URL
type TwitchStreamConfig struct {
	enableTwitchStream *distconf.Bool
	twitchStreamHost   *distconf.Str
	krakenClientID     *distconf.Str
}

// Load TwitchStreamConfig from distconf
func (tc *TwitchStreamConfig) Load(dconf *distconf.Distconf) error {
	tc.enableTwitchStream = dconf.Bool("shine.enable_twitch_stream", true)
	tc.twitchStreamHost = dconf.Str("shine.twitch_stream_host", "")
	tc.krakenClientID = dconf.Str("kraken_client_id", "")
	return nil
}

// TwitchStream handles embed requests
type TwitchStream struct {
	Config *TwitchStreamConfig
	NewReq clients.NewHTTPRequest
	Client clients.RequestDoer
}

// Matches confirms the passed-in url matches the provider
func (t *TwitchStream) Matches(embedURL string) bool {
	if !t.Config.enableTwitchStream.Get() {
		return false
	}
	for _, regexURL := range twitchStreamProviderList {
		if regexURL.MatchString(embedURL) {
			return true
		}
	}
	return false
}

// EntityForURL returns entity derived from embedURL
// EntityForURL returns an error if embedURL can't be converted to an entity
func (t *TwitchStream) EntityForURL(embedURL string) (entity.Entity, error) {
	if !t.Matches(embedURL) {
		return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
	}

	if streamID := t.getStreamID(embedURL); streamID != "" {
		return entity.New(entity.NamespaceStream, streamID), nil
	}
	return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
}

// getStreamID returns the stream channel ID from the embedURL
func (t *TwitchStream) getStreamID(embedURL string) string {
	matches := twitchStreamRegex.FindStringSubmatch(embedURL)
	if len(matches) > 1 {
		return matches[1]
	}
	return ""
}

func (t *TwitchStream) getStreamResponse(ctx context.Context, embedURL string) (*StreamResponse, error) {
	var stream StreamResponse
	channelID := t.getStreamID(embedURL)
	path := "/v5/streams/" + channelID
	err := clients.DoHTTP(ctx, t.Client, "GET", t.Config.twitchStreamHost.Get()+path, nil, nil, &stream, t.twitchAPINewHTTPRequest)
	return &stream, err
}

// RequestEmbed requests embed information for the passed-in embed URL
func (t *TwitchStream) RequestEmbed(ctx context.Context, embedURL string, autoplay bool) (*api.Embed, error) {
	stream, err := t.getStreamResponse(ctx, embedURL)
	if err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	embed := convertStreamToEmbed(stream)
	return sanitizeEmbed(embed, true, t, embedURL, autoplay)
}

// RequestOembed requests oembed information for the passed-in url
func (t *TwitchStream) RequestOembed(ctx context.Context, embedURL string, autoplay bool) (*api.Oembed, error) {
	stream, err := t.getStreamResponse(ctx, embedURL)
	if err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	oembed := convertStreamToOembed(stream)
	return sanitizeOembed(oembed, true, t.Matches, embedURL, autoplay)
}

func (t *TwitchStream) twitchAPINewHTTPRequest(ctx context.Context, method string, url string, body io.Reader) (*http.Request, error) {
	req, err := http.NewRequest(method, url, body)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Client-ID", t.Config.krakenClientID.Get())
	req = req.WithContext(ctx)
	return req, nil
}

func convertStreamToEmbed(stream *StreamResponse) *api.Embed {
	playerHTML := fmt.Sprintf(streamHTMLIframe, stream.Channel.AuthorLogin)
	authorID := strconv.Itoa(stream.Channel.ChannelID)
	if authorID == "0" {
		authorID = ""
	}
	contentID := strconv.Itoa(stream.Stream.BroadcastID)
	if contentID == "0" {
		contentID = ""
	}
	return &api.Embed{
		Type:            "video",
		TwitchType:      entity.NamespaceStream,
		ProviderName:    "Twitch",
		PlayerHTML:      playerHTML,
		Width:           streamHTMLWidth,
		Height:          streamHTMLHeight,
		Title:           stream.Channel.Status,
		AuthorName:      stream.Channel.DisplayName,
		AuthorURL:       stream.Channel.AuthorURL,
		AuthorThumbnail: stream.Channel.AuthorThumbnail,
		AuthorID:        authorID,
		AuthorLogin:     stream.Channel.AuthorLogin,
		ThumbnailURL:    stream.Stream.Thumbnails.Large,
		Thumbnails: &api.Thumbnails{
			Medium: stream.Stream.Thumbnails.Medium,
			Small:  stream.Stream.Thumbnails.Small,
		},
		CreatedAt:       stream.Stream.CreatedAt,
		Game:            stream.Stream.Game,
		ViewCount:       stream.Stream.Viewers,
		StreamType:      stream.Stream.StreamType,
		TwitchContentID: contentID,
	}
}

func convertStreamToOembed(stream *StreamResponse) *api.Oembed {
	playerHTML := fmt.Sprintf(streamHTMLIframe, stream.Channel.AuthorLogin)
	return &api.Oembed{
		Type:         "video",
		Version:      "1.0",
		ProviderName: "Twitch",
		Title:        stream.Channel.Status,
		AuthorName:   stream.Channel.DisplayName,
		AuthorURL:    stream.Channel.AuthorURL,

		HTML:   playerHTML,
		Width:  streamHTMLWidth,
		Height: streamHTMLHeight,
	}
}
