package provider

import (
	"fmt"
	"io"
	"net/http"
	"regexp"
	"strconv"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"golang.org/x/net/context"
)

const (
	vodHTMLWidth  = 500
	vodHTMLHeight = 281
)

var (
	twitchVOD             = regexp.MustCompile(`^(?:https?://)?(?:www\.)?twitch\.tv/videos/([0-9]+)`)
	twitchVODOld          = regexp.MustCompile(`^(?:https?://)?(?:www\.)?twitch\.tv/\w*/v/([0-9]+)`)
	twitchPlayer          = regexp.MustCompile(`^(?:https?://)?player\.twitch\.tv/\?(?:video=v([a-zA-Z0-9]+)|!branding|autoplay=(?:false|true))`)
	twitchVODProviderList = []*regexp.Regexp{twitchVOD, twitchVODOld, twitchPlayer}
	iframeVODHTML         = `<iframe src="https://player.twitch.tv/?video=%s&!branding&autoplay=%s" width="500" height="281" frameborder="0" scrolling="no" allowfullscreen></iframe>`
)

// VODResponse is the response struct for GET VOD
type VODResponse struct {
	TwitchContentID string        `json:"_id"`
	Title           string        `json:"title"`
	Description     string        `json:"description"`
	Thumbnails      VODThumbnails `json:"preview"`
	CreatedAt       *time.Time    `json:"created_at"`
	Game            string        `json:"game"`
	VideoLength     float64       `json:"length"`
	ViewCount       int64         `json:"views"`
	Channel         VODChannel    `json:"channel"`
}

// VODThumbnails is the thumbnail struct for GET VOD response
type VODThumbnails struct {
	Large  string `json:"large"`
	Medium string `json:"medium"`
	Small  string `json:"small"`
}

// VODChannel is the channel struct for GET VOD response
type VODChannel struct {
	DisplayName     string `json:"display_name"`
	URL             string `json:"url"`
	AuthorThumbnail string `json:"logo"`
	AuthorID        int    `json:"_id"`
	AuthorLogin     string `json:"name"`
}

// TwitchVODConfig configures the embed request URL
type TwitchVODConfig struct {
	enableTwitchVOD *distconf.Bool
	twitchVODHost   *distconf.Str
	krakenClientID  *distconf.Str
}

// Load TwitchVODConfig from distconf
func (tc *TwitchVODConfig) Load(dconf *distconf.Distconf) error {
	tc.enableTwitchVOD = dconf.Bool("shine.enable_twitch_vods", false)
	tc.twitchVODHost = dconf.Str("shine.twitch_vod_host", "")
	tc.krakenClientID = dconf.Str("kraken_client_id", "")
	return nil
}

// TwitchVOD handles embed requests
type TwitchVOD struct {
	Config *TwitchVODConfig
	NewReq clients.NewHTTPRequest
	Client clients.RequestDoer
}

// Matches confirms the passed-in url matches the provider
func (t *TwitchVOD) Matches(embedURL string) bool {
	if !t.Config.enableTwitchVOD.Get() {
		return false
	}
	for _, regexURL := range twitchVODProviderList {
		if regexURL.MatchString(embedURL) {
			return true
		}
	}
	return false
}

// EntityForURL returns entity derived from embedURL
// EntityForURL returns an error if embedURL can't be converted to an entity
func (t *TwitchVOD) EntityForURL(embedURL string) (entity.Entity, error) {
	if !t.Matches(embedURL) {
		return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
	}

	if vodID := t.getVODID(embedURL); vodID != "" {
		return entity.New(entity.NamespaceVod, vodID), nil
	}
	return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
}

// getVODID returns the VOD ID from the embed URL
func (t *TwitchVOD) getVODID(embedURL string) string {
	for _, regexURL := range twitchVODProviderList {
		if regexURL.MatchString(embedURL) {
			matches := regexURL.FindStringSubmatch(embedURL)
			if len(matches) > 1 {
				return matches[1]
			}
		}
	}
	return ""
}

func (t *TwitchVOD) getVODResponse(ctx context.Context, embedURL string) (*VODResponse, error) {
	var vod VODResponse
	ID := t.getVODID(embedURL)
	path := "/v5/videos/" + ID
	err := clients.DoHTTP(ctx, t.Client, "GET", t.Config.twitchVODHost.Get()+path, nil, nil, &vod, t.twitchAPINewHTTPRequest)
	return &vod, err
}

// RequestEmbed requests embed information for the passed-in url
func (t *TwitchVOD) RequestEmbed(ctx context.Context, embedURL string, autoplay bool) (*api.Embed, error) {
	vod, err := t.getVODResponse(ctx, embedURL)
	if err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	embed := convertVODToEmbed(vod, embedURL, autoplay)
	return sanitizeEmbed(embed, true, t, embedURL, autoplay)
}

// RequestOembed requests oembed information for the passed-in url
func (t *TwitchVOD) RequestOembed(ctx context.Context, embedURL string, autoplay bool) (*api.Oembed, error) {
	vod, err := t.getVODResponse(ctx, embedURL)
	if err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	oembed := convertVODToOembed(vod, embedURL, autoplay)
	return sanitizeOembed(oembed, true, t.Matches, embedURL, autoplay)
}

func (t *TwitchVOD) twitchAPINewHTTPRequest(ctx context.Context, method string, url string, body io.Reader) (*http.Request, error) {
	req, err := http.NewRequest(method, url, body)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Client-ID", t.Config.krakenClientID.Get())
	req = req.WithContext(ctx)
	return req, nil
}

func convertVODToEmbed(vod *VODResponse, embedURL string, autoplay bool) *api.Embed {
	playerHTML := fmt.Sprintf(iframeVODHTML, vod.TwitchContentID, strconv.FormatBool(autoplay))
	authorID := strconv.Itoa(vod.Channel.AuthorID)
	if authorID == "0" {
		authorID = ""
	}
	return &api.Embed{
		Type:            "video",
		TwitchType:      entity.NamespaceVod,
		ProviderName:    "Twitch",
		RequestURL:      embedURL,
		PlayerHTML:      playerHTML,
		Width:           vodHTMLWidth,
		Height:          vodHTMLHeight,
		Title:           vod.Title,
		AuthorName:      vod.Channel.DisplayName,
		AuthorURL:       vod.Channel.URL,
		AuthorThumbnail: vod.Channel.AuthorThumbnail,
		AuthorID:        authorID,
		AuthorLogin:     vod.Channel.AuthorLogin,
		ThumbnailURL:    vod.Thumbnails.Large,
		Thumbnails: &api.Thumbnails{
			Medium: vod.Thumbnails.Medium,
			Small:  vod.Thumbnails.Small,
		},
		CreatedAt:       vod.CreatedAt,
		Game:            vod.Game,
		VideoLength:     int(vod.VideoLength),
		ViewCount:       int(vod.ViewCount),
		TwitchContentID: vod.TwitchContentID,
	}
}

func convertVODToOembed(vod *VODResponse, embedURL string, autoplay bool) *api.Oembed {
	playerHTML := fmt.Sprintf(iframeVODHTML, vod.TwitchContentID, strconv.FormatBool(autoplay))
	return &api.Oembed{
		Type:         "video",
		Version:      "1.0",
		Title:        vod.Title,
		AuthorName:   vod.Channel.DisplayName,
		AuthorURL:    vod.Channel.URL,
		ProviderName: "Twitch",

		HTML:   playerHTML,
		Width:  vodHTMLWidth,
		Height: vodHTMLHeight,
	}
}
