package provider

import (
	"fmt"
	"net/url"
	"regexp"
	"strconv"
	"strings"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"golang.org/x/net/html"
)

var (
	iframeSrcPlayer = regexp.MustCompile(`^(?:https?:\/\/)?player\.twitch\.tv\/`)
	gifRegex        = regexp.MustCompile(`\.gif$`)
)

const (
	defautIframeHTML         = `<iframe src="%s" width="500" height="281" frameborder="0" scrolling="no" allowfullscreen></iframe>`
	defaultEmbedlyDisplayURL = `https://i.embed.ly/1/display?url=%s&key=%s`
)

type codedError interface {
	HTTPCode() int
}

func getTwitchEntity(embed *api.Embed, p api.Provider, embedURL string) {
	e, err := p.EntityForURL(embedURL)
	if err == nil {
		embed.TwitchEntity = e.Encode()
	}
}

func sanitizeEmbed(embed *api.Embed, allowVideo bool, p api.Provider, embedURL string, autoplay bool) (*api.Embed, error) {
	getTwitchEntity(embed, p, embedURL)

	var err error
	if embed.ProviderName == "Twitch" {
		embed.ProviderURL = "https://www.twitch.tv/"
	}
	if embed.RequestURL == "" && embed.TwitchType != "live" {
		embed.RequestURL = embedURL
	}
	if embed.Type == "video" {
		if isGIF, urlParams := resourceIsGIF(embed.ThumbnailURL); isGIF {
			embed.ThumbnailURL = generateGIFThumbnail(urlParams)
			embed.Type = "gif"
			allowVideo = true
		}
		if !allowVideo {
			return convertVideoEmbedToLinkEmbed(embed), nil
		}
		if embed.TwitchType != entity.NamespaceClip {
			embed.PlayerHTML, err = sanitizeIframe(embed.PlayerHTML, p.Matches, autoplay)
			if err != nil {
				return nil, err
			}
		}
	}
	if embed.Type == "photo" {
		// Set ThumbnailURL to URL value because embedly assigns requested thumbnail image to URL
		embed.ThumbnailURL = embed.URL
	}
	return embed, nil
}

func sanitizeOembed(oembed *api.Oembed, allowVideo bool, matches func(string) bool, oembedURL string, autoplay bool) (*api.Oembed, error) {
	var err error

	switch oembed.Type {
	case "video":
		if isGIF, urlParams := resourceIsGIF(oembed.ThumbnailURL); isGIF {
			oembed.ThumbnailURL = generateGIFThumbnail(urlParams)
			oembed.Type = "gif"
			allowVideo = true
		}
		if !allowVideo {
			// Convert to Link Oembed
			oembed.HTML = ""
			oembed.Type = "link"
			return oembed, nil
		}

		oembed.HTML, err = sanitizeIframe(oembed.HTML, matches, autoplay)
		if err != nil {
			return nil, err
		}
	case "photo":
		// Set ThumbnailURL to URL value because embedly assigns requested thumbnail image to URL
		oembed.ThumbnailURL = oembed.URL
	default:
	}
	return oembed, nil
}

// Returns a sanitized iframe string
func sanitizeIframe(playerHTML string, matches func(string) bool, autoplay bool) (string, error) {
	tokenizer := html.NewTokenizer(strings.NewReader(playerHTML))
	tokens, src := parseTokens(tokenizer)
	src = addQueryParams(src, autoplay)
	if err := verifyTokens(tokens, src, matches); err != nil {
		return "", err
	}
	cleanIframe := fmt.Sprintf(defautIframeHTML, html.EscapeString(src))
	return cleanIframe, nil
}

// Returns all HTML tokens from the tokenizer & the iframe source attribute
func parseTokens(tokenizer *html.Tokenizer) ([]*html.Token, string) {
	tokens := []*html.Token{}
	src := string("")
	for {
		tokenType := tokenizer.Next()
		token := tokenizer.Token()
		if tokenType == html.ErrorToken {
			break
		}
		if tokenType == html.StartTagToken {
			for _, attr := range token.Attr {
				if attr.Key == "src" {
					src = attr.Val
				}
			}
		}
		tokens = append(tokens, &token)
	}
	return tokens, src
}

// Verifies the validity of the HTML content
func verifyTokens(tokens []*html.Token, src string, matches func(string) bool) error {
	if len(tokens) != 2 {
		return errors.New("player html was not recognized")
	}
	if tokens[0].Data != "iframe" && tokens[1].Data != "iframe" {
		return errors.New("player html was not recognized")
	}
	if tokens[0].Type != html.StartTagToken || tokens[1].Type != html.EndTagToken {
		return errors.New("player html was not recognized")
	}
	unescapedURL, err := url.PathUnescape(src)
	if err != nil {
		return errors.Wrap(err, "cannot unescape iframe source URL")
	}
	if !matches(unescapedURL) && !iframeSrcPlayer.MatchString(unescapedURL) {
		return errors.New("embed source was not recognized")
	}
	return nil
}

// Converts a video embed to a link embed
func convertVideoEmbedToLinkEmbed(embed *api.Embed) *api.Embed {
	embed.PlayerHTML = ""
	embed.Type = "link"
	return embed
}

// Adds query params to the video source URL
func addQueryParams(src string, autoplay bool) string {
	u, err := url.Parse(src)
	if err != nil {
		return src
	}
	q := u.Query()
	if q.Get("autoplay") == "" {
		q.Add("autoplay", strconv.FormatBool(autoplay))
	}
	u.RawQuery = q.Encode()
	return u.String()
}

// Check if a URL resource is a gif
func resourceIsGIF(src string) (bool, url.Values) {
	resourceURL, err := url.Parse(src)
	if err != nil {
		return false, nil
	}
	query := resourceURL.Query()
	urlSrc := query.Get("url")
	if gifRegex.MatchString(urlSrc) {
		return true, query
	}
	return false, nil
}

// Returns a URL & HTML tag required to render a gif embed
func generateGIFThumbnail(urlParams url.Values) string {
	urlSrc := urlParams.Get("url")
	key := urlParams.Get("key")
	thumbnailURL := fmt.Sprintf(defaultEmbedlyDisplayURL, urlSrc, key)
	thumbnailURL = thumbnailURL + "&animate=false"
	return thumbnailURL
}

// Returns the HTTP status code of an error
func errorCode(err error) int {
	if coded, ok := errors.Cause(err).(codedError); ok {
		return coded.HTTPCode()
	}
	return 0
}
