package provider

import (
	"net/http"
	"net/url"
	"regexp"

	"strconv"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"golang.org/x/net/context"
)

var (
	vimeo             = regexp.MustCompile(`^(https?:\/\/)?(www\.)?vimeo\.com\/`)
	vimeoPlayer       = regexp.MustCompile(`^(https?:\/\/)?player\.vimeo\.com\/`)
	vimeoProviderList = []*regexp.Regexp{vimeo, vimeoPlayer}
)

// VimeoConfig configures the embed request URL
type VimeoConfig struct {
	vimeoOembedHost *distconf.Str
	enableVimeo     *distconf.Bool
}

// Load VimeoConfig from distconf
func (vc *VimeoConfig) Load(dconf *distconf.Distconf) error {
	vc.vimeoOembedHost = dconf.Str("shine.vimeo_oembed_host", "https://vimeo.com")
	vc.enableVimeo = dconf.Bool("shine.enable_vimeo", false)
	return nil
}

// Vimeo handles embed requests
type Vimeo struct {
	Config *VimeoConfig
	NewReq clients.NewHTTPRequest
	Client clients.RequestDoer
}

// Matches confirms the passed-in url matches the provider
func (v *Vimeo) Matches(embedURL string) bool {
	if !v.Config.enableVimeo.Get() {
		return false
	}
	for _, regexURL := range vimeoProviderList {
		if regexURL.MatchString(embedURL) {
			return true
		}
	}
	return false
}

// EntityForURL returns entity derived from embedURL
// EntityForURL returns an error if embedURL can't be converted to an entity
func (v *Vimeo) EntityForURL(embedURL string) (entity.Entity, error) {
	if !v.Matches(embedURL) {
		return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
	}
	return entity.New(entity.NamespaceOembed, embedURL), nil
}

// RequestEmbed requests embed information for the passed-in url
func (v *Vimeo) RequestEmbed(ctx context.Context, embedURL string, autoplay bool) (*api.Embed, error) {
	var response api.Embed
	path := "/api/oembed.json"
	query := url.Values{}
	query.Add("url", embedURL)
	query.Add("autoplay", strconv.FormatBool(autoplay))
	if err := clients.DoHTTP(ctx, &http.Client{}, "GET", v.Config.vimeoOembedHost.Get()+path, query, nil, &response, v.NewReq); err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	response.ProviderURL = "https://vimeo.com/"
	return sanitizeEmbed(&response, true, v, embedURL, autoplay)
}

// RequestOembed requests oembed information for the passed-in url
func (v *Vimeo) RequestOembed(ctx context.Context, oembedURL string, autoplay bool) (*api.Oembed, error) {
	var response api.Oembed
	path := "/api/oembed.json"
	query := url.Values{}
	query.Add("url", oembedURL)
	query.Add("autoplay", strconv.FormatBool(autoplay))
	if err := clients.DoHTTP(ctx, &http.Client{}, "GET", v.Config.vimeoOembedHost.Get()+path, query, nil, &response, v.NewReq); err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}

	return sanitizeOembed(&response, true, v.Matches, oembedURL, autoplay)
}
