package provider

import (
	"net/http"
	"net/url"
	"regexp"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"golang.org/x/net/context"
)

var (
	youtube             = regexp.MustCompile(`^(https?:\/\/)?(www\.)?youtube\.com\/`)
	youtubeAlt          = regexp.MustCompile(`^(https?:\/\/)?youtu\.be\/`)
	youtubeProviderList = []*regexp.Regexp{youtube, youtubeAlt}
)

// YoutubeConfig configures the embed request URL
type YoutubeConfig struct {
	youtubeOembedHost *distconf.Str
	enableYoutube     *distconf.Bool
}

// Load YoutubeConfig from distconf
func (yc *YoutubeConfig) Load(dconf *distconf.Distconf) error {
	yc.youtubeOembedHost = dconf.Str("shine.youtube_oembed_host", "https://www.youtube.com")
	yc.enableYoutube = dconf.Bool("shine.enable_youtube", false)
	return nil
}

// Youtube handles embed requests
type Youtube struct {
	Config *YoutubeConfig
	NewReq clients.NewHTTPRequest
	Client clients.RequestDoer
}

// Matches confirms the passed-in url matches the provider
func (y *Youtube) Matches(embedURL string) bool {
	if !y.Config.enableYoutube.Get() {
		return false
	}
	for _, regexURL := range youtubeProviderList {
		if regexURL.MatchString(embedURL) {
			return true
		}
	}
	return false
}

// EntityForURL returns entity derived from embedURL
// EntityForURL returns an error if embedURL can't be converted to an entity
func (y *Youtube) EntityForURL(embedURL string) (entity.Entity, error) {
	if !y.Matches(embedURL) {
		return entity.Entity{}, api.ErrorURLDoesNotMatchProvider
	}
	return entity.New(entity.NamespaceOembed, embedURL), nil
}

// RequestEmbed requests embed information for the passed-in url
func (y *Youtube) RequestEmbed(ctx context.Context, embedURL string, autoplay bool) (*api.Embed, error) {
	var response api.Embed
	path := "/oembed"
	query := url.Values{}
	query.Add("url", embedURL)
	if err := clients.DoHTTP(ctx, &http.Client{}, "GET", y.Config.youtubeOembedHost.Get()+path, query, nil, &response, y.NewReq); err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		// Some youtube videos are blocked from embedding (usually this is set by their owners). Youtube returns a
		// 401 in these situations. We return nil, nil to cache the response and return an empty embed object
		if errorCode(err) == http.StatusUnauthorized {
			return nil, nil
		}
		return nil, err
	}
	response.ProviderURL = "https://www.youtube.com/"
	return sanitizeEmbed(&response, true, y, embedURL, autoplay)
}

// RequestOembed requests embed information for the passed-in url
func (y *Youtube) RequestOembed(ctx context.Context, oembedURL string, autoplay bool) (*api.Oembed, error) {
	var response api.Oembed
	path := "/oembed"
	query := url.Values{}
	query.Add("url", oembedURL)
	if err := clients.DoHTTP(ctx, &http.Client{}, "GET", y.Config.youtubeOembedHost.Get()+path, query, nil, &response, y.NewReq); err != nil {
		if errorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		// Some youtube videos are blocked from embedding (usually this is set by their owners). Youtube returns a
		// 401 in these situations. We return nil, nil to cache the response and return an empty embed object
		if errorCode(err) == http.StatusUnauthorized {
			return nil, nil
		}
		return nil, err
	}
	return sanitizeOembed(&response, true, y.Matches, oembedURL, autoplay)
}
