package main

import (
	"expvar"
	"net"
	"net/http"
	"os"
	"os/signal"

	"code.justin.tv/feeds/ctxlog"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"code.justin.tv/feeds/metrics/statsdim"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedcache/gorediscache"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"code.justin.tv/feeds/shine/cmd/shine/internal/provider"
	"code.justin.tv/feeds/spade"
	"github.com/go-redis/redis"
)

const (
	teamName    = "feeds"
	serviceName = "shine"
)

// CodeVersion is set by build script
var CodeVersion string

func sfxStastdConfig() sfxstatsd.SetupConfig {
	return sfxstatsd.SetupConfig{
		CounterMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "http.*.code.*",
				DimensionsMap: "-.handler.-.http_code",
				MetricName:    "http.hits",
			},
			{
				MetricPath:    "redis.cache.*",
				DimensionsMap: "%.%.cache_action",
			},
		},
		TimingMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "http.*.time",
				DimensionsMap: "%.handler.%",
			},
			{
				MetricPath:    "http.get_embed.stats.*.*.duration",
				DimensionsMap: "-.%.%.provider.is_cached.%",
			},
		},
	}
}

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion:    CodeVersion,
		SfxSetupConfig: sfxStastdConfig(),
	},
}

type injectables struct {
	Providers []api.Provider
}

type service struct {
	injectables
	osExit        func(code int)
	onListen      func(net.Addr)
	sigChan       chan os.Signal
	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner
	redisClient   *redis.ClusterClient
	spadeClient   *spade.Client

	server api.HTTPServer

	configs struct {
		HTTPConfig             api.HTTPConfig
		TwitchClipConfig       provider.TwitchClipConfig
		TwitchVODConfig        provider.TwitchVODConfig
		TwitchStreamConfig     provider.TwitchStreamConfig
		TwitchCollectionConfig provider.TwitchCollectionConfig
		TwitchLTVConfig        provider.TwitchLTVConfig
		YoutubeConfig          provider.YoutubeConfig
		EmbedlyConfig          provider.EmbedlyConfig
		VimeoConfig            provider.VimeoConfig
		TwitchEventConfig      provider.TwitchEventConfig
		RedisConfig            shineRedisConfig
		spadeConfig            spade.Config
	}
}

type shineRedisConfig struct {
	gorediscache.GoredisConfig
	addr *distconf.Str
}

func (r *shineRedisConfig) Load(d *distconf.Distconf) error {
	r.addr = d.Str("shine.redis.address", "")
	if r.addr.Get() == "" {
		return errors.New("unable to find shine.redis.addr")
	}
	return r.GoredisConfig.Verify("shine", d)
}

// ClusterOptions gets a redis cluster setup that points to the correct Addrs
func (r *shineRedisConfig) ClusterOptions() *redis.ClusterOptions {
	// Note: We don't update the address if it changes.  The client currently does not support this.
	//       To pick up those changes, we will need to restart the service.
	return &redis.ClusterOptions{
		Addrs: []string{r.addr.Get()},
	}
}

type goredisExpvar struct {
	client *redis.ClusterClient
}

func (g *goredisExpvar) Var() expvar.Var {
	return expvar.Func(func() interface{} {
		return map[string]string{
			"clients:":     g.client.ClientList().String(),
			"cluster_info": g.client.ClusterInfo().String(),
		}
	})
}

func (f *service) setupRedis() error {
	opts := f.configs.RedisConfig.ClusterOptions()
	f.redisClient = redis.NewClusterClient(opts)
	err := f.redisClient.Ping().Err()
	if err != nil {
		return errors.Wrap(err, "unable to ping redis host")
	}
	return nil
}

func (f *service) setup() error {
	if err := f.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(
		f.serviceCommon.Config,
		&f.configs.HTTPConfig,
		&f.configs.TwitchClipConfig,
		&f.configs.TwitchVODConfig,
		&f.configs.TwitchEventConfig,
		&f.configs.TwitchStreamConfig,
		&f.configs.TwitchCollectionConfig,
		&f.configs.TwitchLTVConfig,
		&f.configs.EmbedlyConfig,
		&f.configs.YoutubeConfig,
		&f.configs.VimeoConfig,
		&f.configs.RedisConfig,
		&f.configs.spadeConfig); err != nil {
		return err
	}
	if err := f.setupRedis(); err != nil {
		return err
	}
	return nil
}

func (f *service) inject() {
	httpClient := f.serviceCommon.XRay.Client(&http.Client{})
	twitchProviders := f.Providers
	if f.Providers == nil {
		ctxClient := ctxlog.LoggedDoer{
			Logger: f.serviceCommon.Log,
			C:      &f.serviceCommon.Ctxlog,
			Client: httpClient,
		}
		embedlyProvider := api.Provider(&provider.Embedly{
			Config: &f.configs.EmbedlyConfig,
			NewReq: ctxlog.WrapHTTPRequestWithCtxlog(&f.serviceCommon.Ctxlog, http.NewRequest),
			Client: &ctxClient,
		})
		youtubeProvider := api.Provider(&provider.Youtube{
			Config: &f.configs.YoutubeConfig,
			NewReq: ctxlog.WrapHTTPRequestWithCtxlog(&f.serviceCommon.Ctxlog, http.NewRequest),
			Client: &ctxClient,
		})
		vimeoProvider := api.Provider(&provider.Vimeo{
			Config: &f.configs.VimeoConfig,
			NewReq: ctxlog.WrapHTTPRequestWithCtxlog(&f.serviceCommon.Ctxlog, http.NewRequest),
			Client: &ctxClient,
		})
		twitchEventProvider := api.Provider(&provider.TwitchEvent{
			Config: &f.configs.TwitchEventConfig,
			NewReq: ctxlog.WrapHTTPRequestWithCtxlog(&f.serviceCommon.Ctxlog, http.NewRequest),
			Client: &ctxClient,
		})
		twitchClipProvider := api.Provider(&provider.TwitchClip{
			Config: &f.configs.TwitchClipConfig,
			NewReq: ctxlog.WrapHTTPRequestWithCtxlog(&f.serviceCommon.Ctxlog, http.NewRequest),
			Client: &ctxClient,
		})
		twitchVODProvider := api.Provider(&provider.TwitchVOD{
			Config: &f.configs.TwitchVODConfig,
			NewReq: ctxlog.WrapHTTPRequestWithCtxlog(&f.serviceCommon.Ctxlog, http.NewRequest),
			Client: &ctxClient,
		})
		twitchStreamProvider := api.Provider(&provider.TwitchStream{
			Config: &f.configs.TwitchStreamConfig,
			NewReq: ctxlog.WrapHTTPRequestWithCtxlog(&f.serviceCommon.Ctxlog, http.NewRequest),
			Client: &ctxClient,
		})
		twitchCollectionProvider := api.Provider(&provider.TwitchCollection{
			Config: &f.configs.TwitchCollectionConfig,
			NewReq: ctxlog.WrapHTTPRequestWithCtxlog(&f.serviceCommon.Ctxlog, http.NewRequest),
			Client: &ctxClient,
		})
		twitchLTVProvider := api.Provider(&provider.TwitchLTV{
			Config: &f.configs.TwitchLTVConfig,
		})
		twitchProviders = []api.Provider{twitchEventProvider, twitchClipProvider, twitchVODProvider, twitchStreamProvider, twitchCollectionProvider, twitchLTVProvider}
		f.Providers = append(twitchProviders, embedlyProvider, youtubeProvider, vimeoProvider)
	}
	redisStats := &service_common.StatSender{
		SubStatter:   f.serviceCommon.Statsd.NewSubStatter("redis"),
		ErrorTracker: &f.serviceCommon.ErrorTracker,
	}
	cacheClient := gorediscache.NewGoredisCache(f.redisClient, &f.configs.RedisConfig.GoredisConfig, f.serviceCommon.Environment+":shine", f.serviceCommon.Log, redisStats)
	f.spadeClient = &spade.Client{
		Config:     &f.configs.spadeConfig,
		Logger:     f.serviceCommon.Log,
		HTTPClient: httpClient,
	}
	f.spadeClient.Setup()
	f.server = api.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &f.configs.HTTPConfig.BaseHTTPServerConfig,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
			Dims:        &f.serviceCommon.CtxDimensions,
			Log:         f.serviceCommon.Log,
			ElevateKey:  f.serviceCommon.ElevateLogKey,
			Ctxlog:      &f.serviceCommon.Ctxlog,
			OnListen:    f.onListen,
			PanicLogger: f.serviceCommon.PanicLogger,
			XRay:        f.serviceCommon.XRay,
		},
		Config:           &f.configs.HTTPConfig,
		SpadeClient:      f.spadeClient,
		Redis:            cacheClient,
		DefaultProviders: f.Providers,
		TwitchProviders:  twitchProviders,
	}
	f.server.BaseHTTPServer.SetupRoutes = f.server.SetupRoutes
	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.server, &f.serviceCommon, f.spadeClient,
		},
		SigChan:      f.sigChan,
		SignalNotify: signal.Notify,
	}
	f.serviceCommon.ExpvarHandler.Exported["cluster"] = (&goredisExpvar{client: f.redisClient}).Var()
	// Cannot just pass f because f contains private members that I cannot nil check via reflection
	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"aws-sdk-go", "net/http"},
	}).Check(f, &f.server, f.configs, f.runner, f.Providers, f.spadeClient)
	res.MustBeEmpty(os.Stderr)
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		f.osExit(1)
		return
	}
	f.inject()
	if err := f.runner.Execute(); err != nil {
		f.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished main")
}

func main() {
	instance.main()
}
