// +build integration

package main

import (
	"fmt"
	"net"
	"net/http"
	"os"
	"strconv"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api"
	"code.justin.tv/feeds/shine/cmd/shine/internal/api/mocks"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

const (
	timePerTest = time.Second * 5
)

var (
	validClipEntity = entity.New(entity.NamespaceClip, "SneakyUnusualBeefCclamChamp").Encode()
	validVODEntity  = entity.New(entity.NamespaceVod, "43848179").Encode()
)

func embed(ts *testSetup, url string, autoplay *bool, providers *string) (*api.Embed, error) {
	path := "/v1/embed_for_url?url=" + url
	if autoplay != nil {
		path += "&autoplay=" + strconv.FormatBool(*autoplay)
	}
	if providers != nil {
		path += "&providers=" + *providers
	}
	var ret api.Embed
	err := request(ts, "GET", path, nil, &ret)
	return &ret, err
}

func getEmbedForEntity(ts *testSetup, ent string, autoplay *bool) (*api.Embed, error) {
	path := "/v1/embed_for_entity?entity=" + ent
	if autoplay != nil {
		path += "&autoplay=" + strconv.FormatBool(*autoplay)
	}
	var ret api.Embed
	err := request(ts, "GET", path, nil, &ret)
	return &ret, err
}

func getOembedsForURLs(ts *testSetup, urls []string) (*api.OembedsForURLs, error) {
	path := "/v1/oembeds_for_urls"
	// TODO: decide if we should expose the request param type from the internal/api pkg
	bulkURLs := struct {
		URLs []string
	}{
		URLs: urls,
	}

	var ret api.OembedsForURLs
	err := request(ts, "POST", path, bulkURLs, &ret)
	return &ret, err
}

func getEntitiesForURLs(ts *testSetup, urls []string) (*api.EntitiesForURLs, error) {
	path := "/v1/entities_for_urls"
	bulkURLs := struct {
		URLs []string
	}{
		URLs: urls,
	}

	var ret api.EntitiesForURLs
	err := request(ts, "POST", path, bulkURLs, &ret)
	return &ret, err
}

func timestamp() string {
	return time.Now().Format(time.RFC3339Nano)
}

func TestIntegration_Embed(t *testing.T) {
	host, cleanShutdown := startServer(t, injectables{})
	if host == "" {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+host, t, func(c C) {
		ts := &testSetup{host: host}
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Should be able to fetch a Twitch Clips embed from clips.twitch.tv/slug", func() {
			autoplay := true
			embed, err := embed(ts, "https://clips.twitch.tv/PoliteTrappedOrcaRiPepperonis", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.TwitchEntity, ShouldEqual, "clip:PoliteTrappedOrcaRiPepperonis")
		})

		Convey("Should be able to fetch a Twitch Clips embed from twitch.tv/channelname/clip/slug", func() {
			embed, err := embed(ts, "https://www.twitch.tv/yoda/clip/PoliteTrappedOrcaRiPepperonis", nil, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.TwitchEntity, ShouldEqual, "clip:PoliteTrappedOrcaRiPepperonis")
		})

		Convey("Setting autoplay to false returns an iframe with autoplay set to false", func() {
			autoplay := false
			embed, err := embed(ts, "https://clips.twitch.tv/PoliteTrappedOrcaRiPepperonis", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.PlayerHTML, ShouldContainSubstring, "autoplay=false")
		})

		Convey("Passing a nil autoplay returns an iframe with autoplay set to true", func() {
			embed, err := embed(ts, "https://clips.twitch.tv/StormyComfortableToadArsonNoSexy", nil, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.PlayerHTML, ShouldContainSubstring, "autoplay=true")
		})

		Convey("Setting autoplay to true returns an iframe with autoplay set to true", func() {
			autoplay := true
			embed, err := embed(ts, "https://clips.twitch.tv/StormyComfortableToadArsonNoSexy", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.PlayerHTML, ShouldContainSubstring, "autoplay=true")
		})

		Convey("Should be able to fetch a Twitch VODs embed", func() {
			autoplay := true
			embed, err := embed(ts, "https://www.twitch.tv/qa_flv_vods_transcoded/v/43848179", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.TwitchEntity, ShouldEqual, "vod:43848179")
		})

		Convey("Should be able to fetch a Twitch Live embed", func() {
			autoplay := false
			embed, err := embed(ts, "https://www.twitch.tv/broadcasts/25464189456/channel/27446517", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.TwitchEntity, ShouldEqual, "stream:27446517")
		})

		Convey("Should be able to fetch a Youtube embed", func() {
			autoplay := true
			embed, err := embed(ts, "https://www.youtube.com/watch?v=9bZkp7q19f0", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.TwitchEntity, ShouldEqual, "oembed:https://www.youtube.com/watch?v=9bZkp7q19f0")
		})

		Convey("Should not be able to fetch a Youtube embed if limited to Twitch Content", func() {
			autoplay := true
			providers := "twitch"
			_, err := embed(ts, "https://www.youtube.com/watch?v=9bZkp7q19f0", &autoplay, &providers)
			So(err, ShouldNotBeNil)
		})

		Convey("Should be able to fetch a Vimeo embed", func() {
			autoplay := true
			embed, err := embed(ts, "https://vimeo.com/188568367", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.TwitchEntity, ShouldEqual, "oembed:https://vimeo.com/188568367")
		})

		Convey("Should be able to fetch an gfycat embed", func() {
			autoplay := true
			embed, err := embed(ts, "https://gfycat.com/VainAlarmedGreathornedowl", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "gif")
			So(embed.TwitchEntity, ShouldEqual, "oembed:https://gfycat.com/VainAlarmedGreathornedowl")
		})

		Convey("Should be able to fetch an imgur embed", func() {
			autoplay := true
			embed, err := embed(ts, "http://imgur.com/gallery/WLmKu", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "photo")
			So(embed.TwitchEntity, ShouldEqual, "oembed:http://imgur.com/gallery/WLmKu")
		})

		Convey("Should be able to fetch a reddit upload embed", func() {
			autoplay := true
			embed, err := embed(ts, "https%3A%2F%2Fi.reddituploads.com%2F5b98791139154b1498ab818ef325cd54%3Ffit%3Dmax%26h%3D1536%26w%3D1536%26s%3D5e06a8653193d1f108d00fd78e64b976", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "photo")
			So(embed.TwitchEntity, ShouldEqual, "oembed:https://i.reddituploads.com/5b98791139154b1498ab818ef325cd54?fit=max&h=1536&w=1536&s=5e06a8653193d1f108d00fd78e64b976")
		})

		Convey("Should be able to fetch a Twitch Event", func() {
			autoplay := true
			embed, err := embed(ts, "https://www.twitch.tv/events/9045", &autoplay, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "event")
			So(embed.TwitchEntity, ShouldEqual, "event:9045")
		})

		Convey("Should not fetch embed for an incorrect url", func() {
			autoplay := true
			_, err := embed(ts, "https://www.phishing.com/imgur.com", &autoplay, nil)
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusUnprocessableEntity)
		})

		Convey("Returns 422 for an invalid resource", func() {
			autoplay := true
			_, err := embed(ts, "https://www.youtube.com/watch?v=YIwA2fdsfdsa_WyE", &autoplay, nil)
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusUnprocessableEntity)
		})

		Convey("Returns 422 for Youtube video that doesn't allow embedding", func() {
			autoplay := true
			_, err := embed(ts, "https://www.youtube.com/watch?v=Bi8ZQ5R8M6Y", &autoplay, nil)
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusUnprocessableEntity)
		})
	})

	cleanShutdown(time.Second * 3)
}

func TestIntegration_Oembeds_For_URLs(t *testing.T) {
	host, cleanShutdown := startServer(t, injectables{})
	if host == "" {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+host, t, func(c C) {
		ts := &testSetup{host: host}
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Should be able to convert urls to oembeds", func() {

			clipURL := "https://clips.twitch.tv/TubularSuccessfulSardineSoonerLater"
			onsiteClipURL := "https://www.twitch.tv/lirik/clip/OutstandingCallousDogTheTarFu"
			youtubeURL := "https://www.youtube.com/watch?v=KnSg7AHOI1Q"
			vodURL := "https://www.twitch.tv/qa_flv_vods_transcoded/v/43848179"
			embedlyURL := "https://imgur.com/gallery/QdA2g"
			vimeoURL := "https://vimeo.com/227690432"
			eventURL := "https://www.twitch.tv/events/71735"
			streamURL := "https://www.twitch.tv/broadcasts/25464189456/channel/27446517"
			bulkURLs := []string{youtubeURL, vimeoURL, embedlyURL, clipURL, onsiteClipURL, vodURL, eventURL, streamURL}
			oembedsForURLs, err := getOembedsForURLs(ts, bulkURLs)
			So(err, ShouldBeNil)
			So(oembedsForURLs, ShouldNotBeNil)

			oembeds := oembedsForURLs.Oembeds
			So(oembeds, ShouldHaveLength, len(bulkURLs))

			wantOembedType := map[string]string{
				embedlyURL: "photo",
				eventURL:   "link",

				youtubeURL:    "video",
				vimeoURL:      "video",
				clipURL:       "video",
				onsiteClipURL: "video",
				vodURL:        "video",
				streamURL:     "video",
			}
			for _, oembed := range oembeds {
				t.Log("On", oembed.URL, oembed.Oembed)
				So(oembed.URL, ShouldNotEqual, "")
				So(oembed.Oembed.Type, ShouldEqual, wantOembedType[oembed.URL])
				So(oembed.Oembed.Version, ShouldEqual, "1.0")
				So(oembed.Oembed.ProviderName, ShouldNotEqual, "")
				So(oembed.Oembed.Title, ShouldNotEqual, "")

				if oembedType := oembed.Oembed.Type; oembedType == "video" || oembedType == "rich" {
					So(oembed.Oembed.HTML, ShouldNotEqual, "")
					So(oembed.Oembed.Height, ShouldNotEqual, "")
					So(oembed.Oembed.Width, ShouldNotEqual, "")
				} else if oembedType == "photo" {
					So(oembed.Oembed.URL, ShouldNotEqual, "")
					So(oembed.Oembed.Height, ShouldNotEqual, "")
					So(oembed.Oembed.Width, ShouldNotEqual, "")
				}

				if oembed.URL == youtubeURL {
					So(oembed.Oembed.ProviderURL, ShouldNotEqual, "")
				}
			}
		})

		Convey("Return 400 if POST request's body has 0 urls", func() {
			_, err := getOembedsForURLs(ts, []string{})
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusBadRequest)
		})

		Convey("Silently drop url if there is no supported provider", func() {
			noProviderOembedURL := "https://www.nosupportedprovider.tv/video/embedattempt"
			oembedsForURLs, err := getOembedsForURLs(ts, []string{noProviderOembedURL})
			So(err, ShouldBeNil)
			So(oembedsForURLs, ShouldNotBeNil)
			So(len(oembedsForURLs.Oembeds), ShouldEqual, 0)
			So(oembedsForURLs, ShouldResemble, &api.OembedsForURLs{})
		})

		Convey("Should remove duplicate urls from request body", func() {
			embedlyURL := "https://imgur.com/gallery/QdA2g"
			bulkURLs := []string{embedlyURL, embedlyURL}

			oembeds, err := getOembedsForURLs(ts, bulkURLs)
			So(err, ShouldBeNil)
			So(oembeds, ShouldNotBeNil)
			So(oembeds.Oembeds, ShouldHaveLength, 1)
		})

		Convey("Should return 413 if requests has more than 100 unique URLs", func() {
			var bulkURLs []string
			for i := 0; i <= 100; i++ {
				bulkURLs = append(bulkURLs, "http://"+strconv.Itoa(i)+".com/")
			}
			_, err := getOembedsForURLs(ts, bulkURLs)
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusRequestEntityTooLarge)
		})
	})

	cleanShutdown(3 * time.Second)
}

func TestIntegration_Entity_For_URLs(t *testing.T) {
	host, cleanShutdown := startServer(t, injectables{})
	if host == "" {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With"+host, t, func(c C) {
		ts := &testSetup{host: host}
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		embedlyURL := "https://imgur.com/gallery/QdA2g"
		clipURL := "https://clips.twitch.tv/TubularSuccessfulSardineSoonerLater"
		eventURL := "https://www.twitch.tv/events/71735"
		streamURL := "https://www.twitch.tv/broadcasts/25464189456/channel/27446517"
		vodURL := "https://www.twitch.tv/qa_flv_vods_transcoded/v/43848179"
		youtubeURL := "https://www.youtube.com/watch?v=eJOUkHH7-JY&list=RDMMpjQyBF2gwjQ&index=16"
		vimeoURL := "https://vimeo.com/227690432"

		wantURLEntities := map[string]api.URLAndEntity{
			embedlyURL: {
				URL:    embedlyURL,
				Entity: entity.New(entity.NamespaceOembed, embedlyURL),
			},
			clipURL: {
				URL:    clipURL,
				Entity: entity.New(entity.NamespaceClip, "TubularSuccessfulSardineSoonerLater"),
			},
			eventURL: {
				URL:    eventURL,
				Entity: entity.New(entity.NamespaceEvent, "71735"),
			},
			streamURL: {
				URL:    streamURL,
				Entity: entity.New(entity.NamespaceStream, "27446517"),
			},
			vodURL: {
				URL:    vodURL,
				Entity: entity.New(entity.NamespaceVod, "43848179"),
			},
			youtubeURL: {
				URL:    youtubeURL,
				Entity: entity.New(entity.NamespaceOembed, youtubeURL),
			},
			vimeoURL: {
				URL:    vimeoURL,
				Entity: entity.New(entity.NamespaceOembed, vimeoURL),
			},
		}

		bulkURLs := make([]string, 0)
		for url := range wantURLEntities {
			bulkURLs = append(bulkURLs, url)
		}

		Convey("Should be able to convert vods, clips, streams and oembed urls to entities", func() {
			entities, err := getEntitiesForURLs(ts, bulkURLs)
			So(err, ShouldBeNil)
			bulkEntities := entities.Entities
			So(len(bulkEntities), ShouldEqual, len(wantURLEntities))

			for _, urlAndEntity := range bulkEntities {
				So(urlAndEntity, ShouldResemble, wantURLEntities[urlAndEntity.URL])
			}
		})

		Convey("Return 400 if POST request's body has 0 urls", func() {
			_, err := getEntitiesForURLs(ts, []string{})
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusBadRequest)
		})

		Convey("Silently drop url if there is no supported provider", func() {
			noProviderEmbedURL := "https://www.nosupportedprovider.tv/video/embedattempt"
			entities, err := getEntitiesForURLs(ts, []string{noProviderEmbedURL})
			So(err, ShouldBeNil)
			So(entities, ShouldNotBeNil)
			So(len(entities.Entities), ShouldEqual, 0)
			So(entities, ShouldResemble, &api.EntitiesForURLs{})
		})
	})

	cleanShutdown(time.Second * 3)
}

func TestIntegration_Embed_For_Entity(t *testing.T) {
	host, cleanShutdown := startServer(t, injectables{})
	if host == "" {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+host, t, func(c C) {
		ts := &testSetup{host: host}
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)

		Convey("Should be able to fetch a Twitch Clips embed", func() {
			autoplay := true
			embed, err := getEmbedForEntity(ts, validClipEntity, &autoplay)

			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.RequestURL, ShouldNotEqual, "")
			So(embed.PlayerHTML, ShouldContainSubstring, "autoplay=true")
		})

		Convey("Should be able to fetch a Twitch Live embed", func() {
			autoplay := true
			embed, err := getEmbedForEntity(ts, "stream:27446517:25464189456", &autoplay)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.TwitchType, ShouldEqual, "stream")
		})

		Convey("Should be able to fetch a Twitch VODs embed", func() {
			autoplay := true
			embed, err := getEmbedForEntity(ts, validVODEntity, &autoplay)

			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.PlayerHTML, ShouldContainSubstring, "autoplay=true")
		})

		Convey("Setting autoplay to false returns an iframe with autoplay set to false", func() {
			autoplay := false
			embed, err := getEmbedForEntity(ts, validClipEntity, &autoplay)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.PlayerHTML, ShouldContainSubstring, "autoplay=false")
		})

		Convey("Passing a nil autoplay returns an iframe with autoplay set to true", func() {
			embed, err := getEmbedForEntity(ts, validClipEntity, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.PlayerHTML, ShouldContainSubstring, "autoplay=true")
		})

		Convey("Should return a 422 if given a url", func() {
			_, err := getEmbedForEntity(ts, "https://vimeo.com/188568367", nil)
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusUnprocessableEntity)
		})

		Convey("Should return a 422 if given an invalid entity", func() {
			_, err := getEmbedForEntity(ts, "post:23434-234423-23234-234342", nil)
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusUnprocessableEntity)
		})
	})

	cleanShutdown(time.Second * 3)
}

func TestIntegration_Embed_Cache_Error(t *testing.T) {
	mockProvider := &apimocks.Provider{}
	host, cleanShutdown := startServer(t, injectables{Providers: []api.Provider{api.Provider(mockProvider)}})
	if host == "" {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+host, t, func(c C) {
		ts := &testSetup{host: host}
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)
		mockProvider.Calls = nil
		mockProvider.ExpectedCalls = nil

		Convey("Cache key/value is not set when an error is returned from RequestEmbed", func() {
			autoplay := true
			timestamp := timestamp()
			url := "https://www.twitch.tv/qa_flv_vods_transcoded/v/" + timestamp
			mockProvider.On("Matches", mock.Anything).Return(true)
			mockProvider.On("RequestEmbed", mock.Anything, mock.Anything, mock.Anything).Return(nil, errors.New("Embed Error"))

			embedData, err := embed(ts, url, &autoplay, nil)
			So(err, ShouldBeNil)
			So(embedData.RequestURL, ShouldEqual, url)
			So(embedData.Title, ShouldEqual, "")
			mockProvider.AssertNumberOfCalls(t, "RequestEmbed", 1)

			newEmbedData, err := embed(ts, url, &autoplay, nil)
			So(err, ShouldBeNil)
			So(embedData.RequestURL, ShouldEqual, url)
			So(newEmbedData.Title, ShouldEqual, "")
			mockProvider.AssertNumberOfCalls(t, "RequestEmbed", 2)
		})

		Convey("Cache key/value sets when Embed data is returned from RequestEmbed", func() {
			autoplay := true
			timestamp := timestamp()
			url := "https://www.twitch.tv/qa_flv_vods_transcoded/v/" + timestamp
			mockProvider.On("Matches", mock.Anything).Return(true)
			mockProvider.On("RequestEmbed", mock.Anything, mock.Anything, mock.Anything).Return(
				&api.Embed{RequestURL: "Testurl.com"}, nil)

			embedData, err := embed(ts, url, &autoplay, nil)
			So(err, ShouldBeNil)
			So(embedData, ShouldNotBeNil)
			mockProvider.AssertNumberOfCalls(t, "RequestEmbed", 1)

			newEmbedData, err := embed(ts, url, &autoplay, nil)
			So(err, ShouldBeNil)
			So(newEmbedData, ShouldNotBeNil)
			mockProvider.AssertNumberOfCalls(t, "RequestEmbed", 1)
		})
	})

	cleanShutdown(time.Second * 3)
}

type codedError interface {
	HTTPCode() int
}

func errorCode(err error) int {
	if coded, ok := errors.Cause(err).(codedError); ok {
		return coded.HTTPCode()
	}
	return 0
}

type testSetup struct {
	ctx        context.Context
	cancelFunc func()
	client     *http.Client
	host       string
}

func (ts *testSetup) Setup() error {
	ts.ctx, ts.cancelFunc = context.WithTimeout(context.Background(), timePerTest)
	ts.client = &http.Client{}
	return nil
}

func request(ts *testSetup, method, url string, body interface{}, into interface{}) error {
	return clients.DoHTTP(ts.ctx, ts.client, method, ts.host+url, nil, body, into, nil)
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

func startServer(t *testing.T, i injectables, configs ...map[string][]byte) (string, func(time.Duration)) {
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		"shine.listen_addr":    []byte(":0"),
		"rollbar.access_token": []byte(""),
		"statsd.hostport":      []byte(""),
		"debug.addr":           []byte(":0"),
		"logging.to_stdout":    []byte("false"),
		"logging.to_stderr":    []byte("true"),
	})
	if err != nil {
		t.Error(err)
		return "", nil
	}
	for _, config := range configs {
		if err := addMapValues(localConf, config); err != nil {
			t.Error(err)
			return "", nil
		}
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		injectables: i,
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				ElevateLogKey: elevateKey,
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: t,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(_ ...interface{}) bool {
					return false
				},
			},
			PanicLogger:    panicPanic{},
			SfxSetupConfig: sfxStastdConfig(),
		},
		sigChan: signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	go func() {
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return "", nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 15):
		t.Error("Took to long to start service")
		return "", nil
	}

	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return addressForIntegrationTests, onFinish
}
