package manager

import (
	"testing"

	"code.justin.tv/systems/sandstorm/testutil"
	"github.com/stretchr/testify/assert"
)

func TestManagerDeleteIntegration(t *testing.T) {
	assert := assert.New(t)
	secretValue := "mySecretValue"

	// manager with real dynamo
	m, err := createTestManager()
	if err != nil {
		t.Error(err)
	}

	resetSecret := func(secretName string) {
		err := m.Delete(secretName)
		assert.Nil(err)
	}

	t.Run("manager.delete integration tests", func(t *testing.T) {
		t.Run("delete should only set tombstone to true and be recoverable", func(t *testing.T) {
			secretName := testutil.GetRandomSecretName()
			defer resetSecret(secretName)

			assert.Nil(m.Post(&Secret{
				Name:      secretName,
				Plaintext: []byte(secretValue),
				Class:     ClassCustomer,
			}))

			secret, err := m.Get(secretName)
			if err != nil {
				t.Fatal(err)
			}
			assert.Nil(err)
			assert.Equal([]byte(secretValue), secret.Plaintext)

			assert.Nil(m.Delete(secret.Name))

			// GetVersion should throw ErrSecretTombstoned
			secretVersion, err := m.GetVersion(secretName, secret.UpdatedAt)
			assert.Equal(ErrSecretTombstoned, err)
			assert.Nil(secretVersion)

			// Tombstone should be removable
			assert.Nil(m.removeTombstone(secretName, secret.UpdatedAt))
			secretVersion, err = m.GetVersion(secretName, secret.UpdatedAt)
			assert.Nil(err)
			assert.Equal(secret, secretVersion)
		})
	})
}
