package manager

import (
	"strconv"
	"testing"

	invMocks "code.justin.tv/systems/sandstorm/inventory/heartbeat/mocks"
	"code.justin.tv/systems/sandstorm/mocks"
	"code.justin.tv/systems/sandstorm/queue"
	queueMocks "code.justin.tv/systems/sandstorm/queue/mocks"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

const testSecretName = "testSecretName"

type managerTest struct {
	m             *Manager
	mockDynamoDB  *mocks.DynamoDBAPI
	mockEnveloper *mocks.Enveloper
	mockInventory *invMocks.API
	mockedQueue   *queueMocks.Queuer
}

func (mt *managerTest) teardown(t *testing.T) {
	assert.True(t, mt.mockDynamoDB.AssertExpectations(t))
	assert.True(t, mt.mockEnveloper.AssertExpectations(t))
	assert.True(t, mt.mockInventory.AssertExpectations(t))
	assert.True(t, mt.mockedQueue.AssertExpectations(t))
}

func newManagerTest(t *testing.T) *managerTest {
	_, err := statsd.NewNoopClient()
	if err != nil {
		t.Fatal(err)
	}

	ddb := new(mocks.DynamoDBAPI)
	q := new(queueMocks.Queuer)
	mockedInventory := new(invMocks.API)

	env := new(mocks.Enveloper)

	m := &Manager{
		Envelope:                env,
		DynamoDB:                ddb,
		inventoryClient:         mockedInventory,
		cache:                   &cache{},
		queue:                   q,
		Logger:                  noopLogger,
		stopListeningForUpdates: make(chan struct{}),
		Config:                  Config{},
	}
	return &managerTest{
		m:             m,
		mockDynamoDB:  ddb,
		mockEnveloper: env,
		mockInventory: mockedInventory,
		mockedQueue:   q,
	}
}

func createMocks(t *testing.T) (*Manager, *mocks.DynamoDBAPI, *mocks.Enveloper) {
	mt := newManagerTest(t)
	return mt.m, mt.mockDynamoDB, mt.mockEnveloper
}

func TestListenForUpdates(t *testing.T) {
	testSecret := newTestSecret()

	t.Run("Deletes the entry from cache if update was recieved.", func(t *testing.T) {
		mgr := newManagerTest(t)
		defer mgr.teardown(t)
		mgr.m.cache = &cache{
			secretMap: make(map[string]*Secret),
		}

		mgr.m.cache.Set(testSecret.Name, testSecret)
		assert.NotNil(t, mgr.m.cache.Get(testSecret.Name))

		queueSecret := &queue.Secret{
			UpdatedAt: struct{ S string }{
				S: strconv.FormatInt(testSecret.UpdatedAt, 10),
			},
			Name: struct{ S string }{
				S: testSecret.Name,
			},
		}
		mgr.mockedQueue.On("PollForSecret", mock.Anything).Return(queueSecret, nil).Once()
		err := mgr.m.listenForSecretUpdates()
		assert.Nil(t, err)
		assert.Nil(t, mgr.m.cache.Get(testSecret.Name))
	})

	t.Run("Does not delete the entry from cache if update was recieved.", func(t *testing.T) {
		mgr := newManagerTest(t)
		defer mgr.teardown(t)
		mgr.m.cache = &cache{
			secretMap: make(map[string]*Secret),
		}

		mgr.m.cache.Set(testSecret.Name, testSecret)
		assert.NotNil(t, mgr.m.cache.Get(testSecret.Name))

		mgr.mockedQueue.On("PollForSecret", mock.Anything).Return(nil, nil).Once()
		err := mgr.m.listenForSecretUpdates()
		assert.Nil(t, err)
		assert.NotNil(t, mgr.m.cache.Get(testSecret.Name))
		mgr.m.cache.Delete(testSecret.Name)
	})

	t.Run("ListenForUpdatesInitializes the cache", func(t *testing.T) {
		mgr := newManagerTest(t)
		mgr.mockedQueue.On("Setup").Return(nil)
		defer mgr.teardown(t)

		err := mgr.m.ListenForUpdates()
		assert.NoError(t, err)
		assert.NotNil(t, mgr.m.cache.secretMap)

		mgr.mockedQueue.On("Delete").Return(nil).Once()
		err = mgr.m.StopListeningForUpdates()
		assert.NoError(t, err)
		assert.Nil(t, mgr.m.cache.secretMap)
	})
}
