package manager

import (
	"encoding/base64"
	"strconv"
	"testing"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	. "github.com/smartystreets/goconvey/convey"
)

type secretTestData struct {
	Error     bool
	Namespace *SecretNamespace
}

var secretNamespaceTestData = map[string]secretTestData{
	"syseng/sandstorm/production/x509": secretTestData{
		Error: false,
		Namespace: &SecretNamespace{
			Team:        "syseng",
			System:      "sandstorm",
			Environment: "production",
			Name:        "x509",
		},
	},
	"web/rails/staging/db/pg": secretTestData{
		Error: false,
		Namespace: &SecretNamespace{
			Team:        "web",
			System:      "rails",
			Environment: "staging",
			Name:        "db/pg",
		},
	},
	"chat/tmi/darklaunch/redis-something-or-other": secretTestData{
		Error: false,
		Namespace: &SecretNamespace{
			Team:        "chat",
			System:      "tmi",
			Environment: "darklaunch",
			Name:        "redis-something-or-other",
		},
	},
	"not/a/legit-namespace": secretTestData{
		Error: true,
	},
	"definitely-not-a-namespace": secretTestData{
		Error: true,
	},
	"台/灣/numba/wan": secretTestData{
		Error: false,
		Namespace: &SecretNamespace{
			Team:        "台",
			System:      "灣",
			Environment: "numba",
			Name:        "wan",
		},
	},
}

func TestSecret(t *testing.T) {
	Convey("Manager Secret Tests", t, func() {
		Convey("ParseSecretNamespace", func() {
			Convey("should correctly parse secret names", func() {
				for name, testData := range secretNamespaceTestData {
					std, parseErr := ParseSecretNamespace(name)
					if testData.Error {
						So(parseErr, ShouldNotBeNil)
						continue
					}
					So(parseErr, ShouldBeNil)
					So(std, ShouldResemble, testData.Namespace)
				}
			})
		})
		Convey("asDynamoSecret", func() {
			Convey("should map all fields", func() {
				s := &Secret{
					Name:           "group/project/env/mySecretName",
					UpdatedAt:      147,
					KeyARN:         "myKeyARN",
					DoNotBroadcast: true,
					Tombstone:      true,
					Class:          ClassPublic,
					key:            []byte("myKey"),
					ciphertext:     []byte("myValue"),
				}

				expected := &dynamoSecret{
					Name:           "group/project/env/mySecretName",
					Namespace:      "group",
					UpdatedAt:      147,
					KeyARN:         "myKeyARN",
					DoNotBroadcast: true,
					Tombstone:      true,
					Class:          1,
					Key:            "bXlLZXk=",
					Value:          "bXlWYWx1ZQ==",
				}

				So(s.asDynamoSecret(), ShouldResemble, expected)
			})
			Convey("should map invalid Classes to ClassDefault", func() {
				for _, class := range []int{-1, 0, 5, 6} {
					s := &Secret{Class: SecretClass(class)}
					expected := &dynamoSecret{Class: int(ClassDefault)}
					So(s.asDynamoSecret(), ShouldResemble, expected)
				}

			})
		})
	})
}

func getValidSecretForUnMarshaling() map[string]*dynamodb.AttributeValue {
	doNotBroadcast := false
	tombstone := false
	keyarn := "somearn"

	secretName, updatedAt, cipherText, key := getSecretRequiredParams()

	validSecret := map[string]*dynamodb.AttributeValue{
		"name":             &dynamodb.AttributeValue{S: &secretName},
		"do_not_broadcast": &dynamodb.AttributeValue{BOOL: &doNotBroadcast},
		"tombstone":        &dynamodb.AttributeValue{BOOL: &tombstone},
		"key_arn":          &dynamodb.AttributeValue{S: &keyarn},
		"updated_at":       &dynamodb.AttributeValue{N: &updatedAt},
		"value":            &dynamodb.AttributeValue{S: &cipherText},
		"key":              &dynamodb.AttributeValue{S: &key},
	}
	return validSecret
}

func getSecretRequiredParams() (string, string, string, string) {
	secretName := "somename"
	updatedAt := "1"
	cipherText := base64.StdEncoding.EncodeToString([]byte("cipherTextValue"))
	key := base64.StdEncoding.EncodeToString([]byte("key"))
	return secretName, updatedAt, cipherText, key
}

func TestUnmarshalSecret(t *testing.T) {
	Convey("Manager Unmarshal JSON", t, func() {
		Convey("should successfully parse valid record", func() {
			secretToUnmarshal := getValidSecretForUnMarshaling()

			secret, err := unmarshalSecret(secretToUnmarshal)
			So(err, ShouldBeNil)
			So(secret, ShouldNotBeNil)
			So(secret.Name, ShouldEqual, "somename")
		})

		Convey("Should throw tombstone error if Tombstone is true", func() {
			secretToUnmarshal := getValidSecretForUnMarshaling()
			*secretToUnmarshal["tombstone"].BOOL = true
			_, err := unmarshalSecret(secretToUnmarshal)
			// So(secret, ShouldBeNil)
			So(err, ShouldEqual, ErrSecretTombstoned)
		})

		Convey("Should throw missing field when one or more required fields are missing", func() {

			secretName, updatedAt, cipherText, key := getSecretRequiredParams()

			invalidJSONs := []map[string]*dynamodb.AttributeValue{
				map[string]*dynamodb.AttributeValue{
					"name": &dynamodb.AttributeValue{S: &secretName},
				},
				map[string]*dynamodb.AttributeValue{
					"name":       &dynamodb.AttributeValue{S: &secretName},
					"updated_at": &dynamodb.AttributeValue{N: &updatedAt},
				},
				map[string]*dynamodb.AttributeValue{
					"name":       &dynamodb.AttributeValue{S: &secretName},
					"updated_at": &dynamodb.AttributeValue{N: &updatedAt},
					"value":      &dynamodb.AttributeValue{S: &cipherText},
				},
				map[string]*dynamodb.AttributeValue{
					"key":        &dynamodb.AttributeValue{S: &key},
					"updated_at": &dynamodb.AttributeValue{N: &updatedAt},
					"value":      &dynamodb.AttributeValue{S: &cipherText},
				},
				map[string]*dynamodb.AttributeValue{},
			}
			for _, secretToUnmarshal := range invalidJSONs {
				secret, err := unmarshalSecret(secretToUnmarshal)
				So(secret, ShouldBeNil)
				So(err, ShouldNotBeEmpty)
				So(err.Error(), ShouldContainSubstring, `value required`)
			}
		})

		Convey("should return class as 'ClassDefault' for a record with a depreciated class.", func() {
			for _, class := range []string{"0", "5"} {
				secretToUnmarshal := getValidSecretForUnMarshaling()
				secretToUnmarshal["class"] = &dynamodb.AttributeValue{N: aws.String(class)}

				secret, err := unmarshalSecret(secretToUnmarshal)
				So(secret, ShouldNotBeNil)
				So(err, ShouldBeNil)
				So(secret.Class, ShouldEqual, ClassDefault)
			}
		})

		Convey("Should set class to specified value", func() {
			secretToUnmarshal := getValidSecretForUnMarshaling()
			class := strconv.Itoa(int(ClassInternal))
			secretToUnmarshal[jsonClassField] = &dynamodb.AttributeValue{N: &class}
			secret, err := unmarshalSecret(secretToUnmarshal)
			So(secret, ShouldNotBeNil)
			So(err, ShouldBeNil)
			Convey("Secret class was set to Confidential", func() {
				So(secret.Class, ShouldEqual, ClassInternal)
			})
		})
	})
}

func TestFillPlaintext(t *testing.T) {
	Convey("TestfillPlaintext", t, func() {
		Convey("Should 20 character plaintext", func() {
			s := Secret{}
			So(s.FillPlaintext(&FillPlaintextRequest{
				Length: 20,
			}), ShouldBeNil)
			So(len(s.Plaintext), ShouldEqual, 28)
			So(s.Plaintext, ShouldNotResemble, make([]byte, 28))
		})

		Convey("Should error out for less than 2", func() {
			s := Secret{}
			So(s.FillPlaintext(&FillPlaintextRequest{
				Length: 1,
			}), ShouldNotBeNil)
		})

		Convey("Should error out for greater than than 4096", func() {
			s := Secret{}
			So(s.FillPlaintext(&FillPlaintextRequest{
				Length: 4097,
			}), ShouldNotBeNil)
		})
	})
}
