package spadetest

import (
	"net/http"
	"net"
	"io"
	"code.justin.tv/feeds/spade"
	"encoding/json"
	"sync"
	"encoding/base64"
	"code.justin.tv/feeds/distconf"
	"fmt"
	"golang.org/x/net/context"
)

type TestingSpadeServer struct {
	server http.Server
	listener net.Listener

	Events chan spade.Event

	listenErr error
	once sync.Once
}

var _ http.Handler = &TestingSpadeServer{}

func (t *TestingSpadeServer) setup() {
	t.once.Do(func() {
		t.listener, t.listenErr = net.Listen("tcp", "0.0.0.0:0")
	})
}

// Drain the outstanding event queue
func (t *TestingSpadeServer) Drain() {
	for {
		select {
		case <- t.Events:
		default:
			return
		}
	}
}

// Find a matching event, waiting till ctx.  If the event is not found, return nil
func (t *TestingSpadeServer) Find(ctx context.Context, f func(e spade.Event) bool) *spade.Event {
	for {
		select {
		case e := <- t.Events:
			if f(e) {
				return &e
			}
		case <- ctx.Done():
			return nil
		}
	}
}

func (t *TestingSpadeServer) SetupDistconf(mem *distconf.InMemory) error {
	if err := mem.Write("spade.endpoint", []byte(t.ListenAddr())); err != nil {
		return err
	}
	return nil
}

func (t *TestingSpadeServer) Start() error {
	t.setup()
	if t.listenErr != nil {
		return t.listenErr
	}
	if t.Events == nil {
		t.Events = make(chan spade.Event, 1024)
	}
	t.server.Handler = t
	return t.server.Serve(t.listener)
}

func (t *TestingSpadeServer) ListenAddr() string {
	t.setup()
	return fmt.Sprintf("http://127.0.0.1:%d", t.listener.Addr().(*net.TCPAddr).Port)
}

func (t *TestingSpadeServer) Close() error {
	t.setup()
	return t.listener.Close()
}

func (t *TestingSpadeServer) ServeHTTP(rw http.ResponseWriter, req *http.Request) {
	if req.URL.Path == "/xarth" {
		io.WriteString(rw, "XARTH")
		rw.WriteHeader(http.StatusOK)
		return
	}
	if req.URL.Path != "/track" {
		rw.WriteHeader(http.StatusNotFound)
		return
	}
	var allEvents []spade.Event
	decodedBytes, err := base64.URLEncoding.DecodeString(req.URL.Query().Get("data"))
	if err != nil {
		rw.WriteHeader(http.StatusBadRequest)
		io.WriteString(rw, err.Error())
		return
	}
	if err := json.Unmarshal(decodedBytes, &allEvents); err != nil {
		rw.WriteHeader(http.StatusBadRequest)
		io.WriteString(rw, err.Error())
		fmt.Println(req.URL.Query().Get("data"))
		return
	}
	for _, e := range allEvents {
		select {
		case t.Events <- e:
			default:
			rw.WriteHeader(http.StatusInternalServerError)

		}

	}
	rw.WriteHeader(http.StatusNoContent)
}