# allow builds
resource "aws_iam_role" "build_deployment" {
  # Note: You will have to run this terraform twice, if you get an -Invalid principal in policy- error
  name               = "${var.team}-${var.environment}-build-deployment"
  assume_role_policy = data.aws_iam_policy_document.build_deployment.json
  description        = "role for deployment to assume when needing to deploy new versions of code"
}

locals {
  assume_role_policies = compact([aws_iam_user.deployment.arn, var.ecs_on_demand_role])
}

data "aws_iam_policy_document" "build_deployment" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs.amazonaws.com"]
    }
  }

  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "AWS"
      identifiers = local.assume_role_policies
    }
  }
}

data "aws_caller_identity" "current" {
}

data "aws_iam_policy_document" "pipeline_ssm_permissions" {
  statement {
    actions   = ["ssm:GetParameters", "ssm:DescribeParameters"]
    resources = ["arn:aws:ssm:*:${data.aws_caller_identity.current.account_id}:parameter/pipeline/${var.team}/*"]
  }
}

resource "aws_iam_policy" "pipeline_ssm_permissions" {
  name        = "${var.team}-${var.environment}-ssm-values"
  description = "Allows promotion script to read pipeline variables"
  policy      = data.aws_iam_policy_document.pipeline_ssm_permissions.json
}

resource "aws_iam_role_policy_attachment" "build_deployment_attach_policy_ssm_permissions" {
  role       = aws_iam_role.build_deployment.name
  policy_arn = aws_iam_policy.pipeline_ssm_permissions.arn
}

# devtools jenkins won't allow me to assume roles directly and requires aws credentials stored on the box.
# This is a security flaw that I disagree with, but I also cannot maintain a jenkins cluster myself, so I will do as
# they request.
resource "aws_iam_user" "deployment" {
  name = "${var.team}-${var.environment}-deploy"
}

data "aws_iam_policy_document" "allow_build_user_assume_build_role" {
  statement {
    actions   = ["sts:AssumeRole"]
    resources = [aws_iam_role.build_deployment.arn]
  }
}

resource "aws_iam_user_policy" "allow_user_assume_role" {
  name   = "${var.team}-${var.environment}-deploy-user-assume-role"
  user   = aws_iam_user.deployment.name
  policy = data.aws_iam_policy_document.allow_build_user_assume_build_role.json
}

// Allows build service to modify EC2 container tasks
resource "aws_iam_role_policy_attachment" "build_deployment_attach_policyEC2" {
  role       = aws_iam_role.build_deployment.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonEC2ContainerServiceFullAccess"
}

// Unsure ... probably unneeded??
resource "aws_iam_role_policy_attachment" "build_deployment_attach_policyS3" {
  role       = aws_iam_role.build_deployment.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess"
}

// Allows deployment to print out service STDOUT
resource "aws_iam_role_policy_attachment" "build_deployment_attach_policyCWlogs" {
  role       = aws_iam_role.build_deployment.name
  policy_arn = "arn:aws:iam::aws:policy/CloudWatchLogsReadOnlyAccess"
}

# Setup IAM to allow services to register w/ the ALB
resource "aws_iam_role" "ecs_service" {
  name               = "${var.team}-${var.environment}-ecs-service"
  assume_role_policy = data.aws_iam_policy_document.ecs_service.json
  description        = "Allows the service to register tasks with the ALB"
}

data "aws_iam_policy_document" "ecs_service" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs.amazonaws.com"]
    }
  }
}

resource "aws_iam_instance_profile" "ecs_service" {
  name = "${var.team}-${var.environment}-ecs-service"
  role = aws_iam_role.ecs_service.name
}

resource "aws_iam_role_policy_attachment" "ecs_service_attach_policy" {
  role       = aws_iam_role.ecs_service.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceRole"
}

# Setup IAM to allow task autoscaling
resource "aws_iam_role" "ecs_autoscale" {
  name               = "${var.team}-${var.environment}-ecs-task-autoscale"
  description        = "Used by AWS's built in task auto scaler to modify running task count on cloudwatch alarms"
  assume_role_policy = data.aws_iam_policy_document.ecs_autoscale.json
}

data "aws_iam_policy_document" "ecs_autoscale" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["application-autoscaling.amazonaws.com"]
    }
  }
}

resource "aws_iam_instance_profile" "ecs_autoscale" {
  name = "${var.team}-${var.environment}-ecs-autoscale"
  role = aws_iam_role.ecs_autoscale.name
}

resource "aws_iam_role_policy_attachment" "ecs_autoscale_attach_policy" {
  role       = aws_iam_role.ecs_autoscale.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceAutoscaleRole"
}

# Setup IAM to allow hosts to update ECS deployed tasks

resource "aws_iam_role" "ecs_host" {
  description        = "Allows EC2 instance to be part of an ECS cluster"
  name               = "${var.team}-${var.environment}-ecs-host"
  assume_role_policy = data.aws_iam_policy_document.ecs_host.json
}

data "aws_iam_policy_document" "ecs_host" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

resource "aws_iam_instance_profile" "ecs_host" {
  name = "${var.team}-${var.environment}-ecs-host"
  role = aws_iam_role.ecs_host.name
}

# This lets the host run the ecs-agent and be part of an ECS cluster
resource "aws_iam_role_policy_attachment" "ecs_host_attach_policy" {
  role       = aws_iam_role.ecs_host.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceforEC2Role"
}

# This allows the host to run the SSM agent, which lets us roll out docker and ecs-agent updates
resource "aws_iam_role_policy_attachment" "ecs_host_ssm_policy" {
  role       = aws_iam_role.ecs_host.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM"
}

### ----- task level iam policies -----
resource "aws_iam_role" "xray_task" {
  name               = "${var.team}-${var.environment}-xray"
  description        = "Allows sending xray stats to the web AWS account"
  assume_role_policy = data.aws_iam_policy_document.xray_task.json
}

data "aws_iam_policy_document" "xray_task" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }
  }
}

resource "aws_iam_role_policy" "policy" {
  name = "${var.team}-${var.environment}-xray-assume-role"
  role = aws_iam_role.xray_task.id

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": "sts:AssumeRole",
      "Resource": ["${var.xray_role}"]
    }
  ]
}
EOF

}

# Output variables
output "container_iam_role" {
  value = aws_iam_role.ecs_host.arn
}

output "service_iam_role" {
  value = aws_iam_role.ecs_service.arn
}

output "xray_task_role" {
  value = aws_iam_role.xray_task.arn
}

output "task_autoscale_iam_role" {
  value = aws_iam_role.ecs_autoscale.arn
}

output "container_instance_profile_id" {
  value = aws_iam_instance_profile.ecs_host.id
}

output "build_role" {
  value = aws_iam_role.build_deployment.arn
}

output "build_user" {
  value = aws_iam_user.deployment.name
}

output "build_user_arn" {
  value = aws_iam_user.deployment.arn
}

