# Setup task for statsite

resource "aws_ecs_task_definition" "statsite" {
  family                = "${var.team}-${var.environment}-statsite"
  container_definitions = data.template_file.statsite_task_def.rendered
  network_mode          = "bridge"
}

data "template_file" "statsite_task_def" {
  template = file("${path.module}/statsite_task_definition.json")

  vars = {
    image    = "docker.pkgs.xarth.tv/statsite"
    region   = var.region
    env      = var.environment
    loggroup = aws_cloudwatch_log_group.container-logs.name
    mem      = 256
    cpu      = 256 // 1/4 a core should be enough ...
  }
}

# Setup task for xray
# TODO: Attack correct xray permissions

data "template_file" "xray_task_def" {
  template = file("${path.module}/xray_task_definition.json")

  vars = {
    image    = "docker.pkgs.xarth.tv/xray"
    region   = var.region
    env      = var.environment
    loggroup = aws_cloudwatch_log_group.container-logs.name
    mem      = 256
    cpu      = 256 // 1/4 a core should be enough ...
    xray_arn = var.xray_web_arn
  }
}

resource "aws_ecs_task_definition" "xray" {
  family                = "${var.team}-${var.environment}-xray"
  container_definitions = data.template_file.xray_task_def.rendered
  network_mode          = "bridge"
  task_role_arn         = var.xray_task_arn
}

# Setup log group for ECS container logs

resource "aws_cloudwatch_log_group" "container-logs" {
  name              = "${var.team}-${var.environment}-container-logs"
  retention_in_days = var.container_log_retention

  tags = {
    env  = var.environment
    team = var.team
  }
}

resource "aws_cloudwatch_log_group" "ssm-logs" {
  name              = "${var.team}-${var.environment}-ssm-logs"
  retention_in_days = var.ssm_log_retention

  tags = {
    env  = var.environment
    team = var.team
  }
}

resource "aws_cloudwatch_log_group" "ecs-agent-logs" {
  name              = "${var.team}-${var.environment}-ecs-agent-logs"
  retention_in_days = var.ssm_log_retention

  tags = {
    env  = var.environment
    team = var.team
  }
}

resource "aws_cloudwatch_log_group" "ecs-init-logs" {
  name              = "${var.team}-${var.environment}-ecs-init-logs"
  retention_in_days = var.ssm_log_retention

  tags = {
    env  = var.environment
    team = var.team
  }
}

module "common_cluster" {
  source               = "../ecscluster"
  environment          = var.environment
  name                 = "${var.team}-${var.environment}-common"
  team                 = var.team
  xray_task_family     = aws_ecs_task_definition.xray.family
  statsite_task_family = aws_ecs_task_definition.statsite.family
}

# Output variables

output "container_loggroup" {
  value = aws_cloudwatch_log_group.container-logs.name
}

output "ssm_loggroup" {
  value = aws_cloudwatch_log_group.ssm-logs.name
}

output "ecs_init_loggroup" {
  value = aws_cloudwatch_log_group.ecs-init-logs.name
}

output "ecs_agent_loggroup" {
  value = aws_cloudwatch_log_group.ecs-agent-logs.name
}

output "container_loggroup_region" {
  value = var.region
}

output "statsite_task_family" {
  value = aws_ecs_task_definition.statsite.family
}

output "xray_task_family" {
  value = aws_ecs_task_definition.xray.family
}

output "common_cluster_name" {
  value = module.common_cluster.cluster_name
}

output "common_cluster_id" {
  value = module.common_cluster.cluster_id
}

