package datapipeline

import (
	"encoding/json"
	"errors"

	"github.com/aws/aws-sdk-go/aws"
	awsPipeline "github.com/aws/aws-sdk-go/service/datapipeline"
	"github.com/hashicorp/terraform/helper/schema"
)

type encodedPipeline struct {
	PipelineObjects  []map[string]interface{} `json:"objects"`
	ParameterObjects []map[string]string      `json:"parameters"`
	ParameterValues  map[string]string        `json:"values"`
}

func (e *encodedPipeline) addParameterObjects(ret *awsPipeline.PutPipelineDefinitionInput) {
	for _, obj := range e.ParameterObjects {
		o := awsPipeline.ParameterObject{}
		for key, val := range obj {
			if key == "id" {
				o.Id = aws.String(val)
				continue
			}
			o.Attributes = append(o.Attributes, &awsPipeline.ParameterAttribute{
				Key:         aws.String(key),
				StringValue: aws.String(val),
			})
		}
		ret.ParameterObjects = append(ret.ParameterObjects, &o)
	}
}

func (e *encodedPipeline) GetPutPipelineDefinitionInput() (*awsPipeline.PutPipelineDefinitionInput, error) {
	if e.PipelineObjects == nil {
		return nil, errors.New("expected objects value")
	}
	ret := &awsPipeline.PutPipelineDefinitionInput{}
	for _, obj := range e.PipelineObjects {
		o := awsPipeline.PipelineObject{}
		for key, val := range obj {
			if key == "id" {
				o.Id = aws.String(val.(string))
				continue
			}
			if key == "name" {
				o.Name = aws.String(val.(string))
				continue
			}
			if valAsStr, ok := val.(string); ok {
				o.Fields = append(o.Fields, &awsPipeline.Field{
					Key:         aws.String(key),
					StringValue: &valAsStr,
				})
				continue
			}
			if valAsMap, ok := val.(map[string]interface{}); ok {
				o.Fields = append(o.Fields, &awsPipeline.Field{
					Key:      aws.String(key),
					RefValue: aws.String(valAsMap["ref"].(string)),
				})
				continue
			}
			if valAsArr, ok := val.([]interface{}); ok {
				fieldsAddSuccess := true
				for _, arrayVal := range valAsArr {
					if valAsMap, ok := arrayVal.(map[string]interface{}); ok {
						o.Fields = append(o.Fields, &awsPipeline.Field{
							Key:      aws.String(key),
							RefValue: aws.String(valAsMap["ref"].(string)),
						})
					} else {
						fieldsAddSuccess = false
					}
				}
				if fieldsAddSuccess {
					continue
				}
			}

			return nil, errors.New("invalid pipeline definition input")
		}
		ret.PipelineObjects = append(ret.PipelineObjects, &o)
	}

	e.addParameterObjects(ret)

	for k, v := range e.ParameterValues {
		ret.ParameterValues = append(ret.ParameterValues, &awsPipeline.ParameterValue{
			Id:          aws.String(k),
			StringValue: aws.String(v),
		})
	}
	return ret, nil
}

type NewResource struct {
	dataPipeline *awsPipeline.DataPipeline
}

func (n *NewResource) DataPipeline(meta interface{}) *awsPipeline.DataPipeline {
	if n.dataPipeline == nil {
		return meta.(*awsPipeline.DataPipeline)
	}
	return n.dataPipeline
}

func (n *NewResource) Resource() *schema.Resource {
	return &schema.Resource{
		Create: n.Create,
		Read:   n.Read,
		Update: n.Update,
		Delete: n.Delete,
		Importer: &schema.ResourceImporter{
			State: schema.ImportStatePassthrough,
		},

		Schema: n.Schema(),
	}
}

func (n *NewResource) Create(d *schema.ResourceData, meta interface{}) error {
	unique_id := d.Get("name").(string)
	if d.Get("unique_id").(string) != "" {
		unique_id = d.Get("unique_id").(string)
	}
	in := &awsPipeline.CreatePipelineInput{
		Name:     aws.String(d.Get("name").(string)),
		UniqueId: aws.String(unique_id),
	}
	desc := d.Get("description")
	if desc != nil {
		in.Description = aws.String(desc.(string))
	}
	out, err := n.DataPipeline(meta).CreatePipeline(in)
	if err != nil {
		return err
	}
	d.SetId(*out.PipelineId)

	v := encodedPipeline{}
	if err = json.Unmarshal([]byte(d.Get("definition").(string)), &v); err != nil {
		return err
	}

	inReq, err := v.GetPutPipelineDefinitionInput()
	if err != nil {
		return err
	}
	inReq.PipelineId = out.PipelineId
	res, err := n.DataPipeline(meta).PutPipelineDefinition(inReq)
	if err != nil {
		return err
	}
	if res.Errored != nil && *res.Errored == false {
		_, err = n.DataPipeline(meta).ActivatePipeline(&awsPipeline.ActivatePipelineInput{
			PipelineId: out.PipelineId,
		})
		return err
	}
	return nil
}

func (n *NewResource) Update(d *schema.ResourceData, meta interface{}) error {
	v := encodedPipeline{}
	if err := json.Unmarshal([]byte(d.Get("definition").(string)), &v); err != nil {
		return err
	}

	inReq, err := v.GetPutPipelineDefinitionInput()
	if err != nil {
		return err
	}
	inReq.PipelineId = aws.String(d.Id())
	res, err := n.DataPipeline(meta).PutPipelineDefinition(inReq)
	if err != nil {
		return err
	}
	// Must activate pipelines that were modified.
	if res.Errored != nil && *res.Errored == false {
		_, err = n.DataPipeline(meta).ActivatePipeline(&awsPipeline.ActivatePipelineInput{
			PipelineId: aws.String(d.Id()),
		})
		return err
	}
	return nil
}

func (n *NewResource) Read(d *schema.ResourceData, meta interface{}) error {
	id := d.Id()
	descIn := &awsPipeline.DescribePipelinesInput{
		PipelineIds: []*string{&id},
	}
	descOut, err := n.DataPipeline(meta).DescribePipelines(descIn)
	if err != nil {
		return err
	}
	if len(descOut.PipelineDescriptionList) != 1 {
		d.SetId("")
		return nil
	}
	pipelineDesc := descOut.PipelineDescriptionList[0]
	d.Set("description", strOrNil(pipelineDesc.Description))
	d.Set("name", strOrNil(pipelineDesc.Name))

	_, err = n.DataPipeline(meta).GetPipelineDefinition(&awsPipeline.GetPipelineDefinitionInput{
		PipelineId: &id,
	})
	if err != nil {
		return err
	}
	return nil
}

func strOrNil(s *string) string {
	if s == nil {
		return ""
	}
	return *s
}

func (n *NewResource) Delete(d *schema.ResourceData, meta interface{}) error {
	in := &awsPipeline.DeletePipelineInput{
		PipelineId: aws.String(d.Id()),
	}
	_, err := n.DataPipeline(meta).DeletePipeline(in)
	return err
}

func (n *NewResource) Schema() map[string]*schema.Schema {
	return map[string]*schema.Schema{
		"name": {
			Type:        schema.TypeString,
			Required:    true,
			Description: "Name of the pipeline",
			ForceNew:    true,
		},
		"unique_id": {
			Type:        schema.TypeString,
			Default:     "",
			Optional:    true,
			Description: "Unique ID of the pipeline.  If not set, will be the pipeline's name.",
			ForceNew:    true,
		},
		"description": {
			Type:        schema.TypeString,
			Optional:    true,
			Description: "The description for the pipeline",
		},
		"definition": {
			Type:        schema.TypeString,
			Required:    true,
			Description: "The pipeline definition to import (found via pipeline export)",
		},
	}
}
