package datapipeline

import (
	"strings"
	"testing"
	"time"

	"code.justin.tv/feeds/errors"
	awsPipeline "github.com/aws/aws-sdk-go/service/datapipeline"
	"github.com/hashicorp/terraform/helper/resource"
	"github.com/hashicorp/terraform/helper/schema"
	"github.com/hashicorp/terraform/terraform"
)

func TestAccAWSCodePipeline_simpleSetup(t *testing.T) {
	p := NewProvider()
	resource.Test(t, resource.TestCase{
		Providers: map[string]terraform.ResourceProvider{
			"datapipeline": p,
		},
		CheckDestroy: checkDestroy(p),
		Steps: []resource.TestStep{
			{
				Config: datapipelineConfig,
				Check: resource.ComposeTestCheckFunc(
					resource.TestCheckResourceAttr("datapipeline.testing", "name", "testing-pipeline-unittest"),
				),
			},
			{
				Config: datapipelineConfigUpdateDef,
				Check:  resource.ComposeTestCheckFunc(),
			},
		},
	})
}

func checkDestroy(p terraform.ResourceProvider) func(s *terraform.State) error {
	return func(s *terraform.State) error {
		removed := false
		for _, rs := range s.RootModule().Resources {
			if rs.Type != "datapipeline" {
				continue
			}
			i := 0
			for i = 0; i < 20; i++ {
				descOut, err := p.(*schema.Provider).Meta().(*awsPipeline.DataPipeline).DescribePipelines(&awsPipeline.DescribePipelinesInput{
					PipelineIds: []*string{&rs.Primary.ID},
				})
				if err != nil {
					if strings.Contains(err.Error(), "does not exist") {
						break
					}
					return err
				}
				if len(descOut.PipelineDescriptionList) == 0 {
					break
				}
				time.Sleep(time.Second)
			}
			if i == 20 {
				return errors.Errorf("pipeline still exists: %s", rs.Primary.ID)
			}
			removed = true
		}
		if !removed {
			return errors.New("could not find pipeline to remove")
		}
		return nil
	}
}

const datapipelineConfig = `
provider "datapipeline" {
  region  = "us-west-2"
  profile = "twitch-feed-dev"
}

resource datapipeline "testing" {
  name      = "testing-pipeline-unittest"

  #  description = "this is testing terraform"
  definition = <<EOF
{
  "objects": [{
      "period": "7 Days",
      "name": "Every 7 days b",
      "id": "DefaultSchedule",
      "type": "Schedule",
      "startAt": "FIRST_ACTIVATION_DATE_TIME"
    }],
  "parameters": [
    {
      "default": "0.25",
      "watermark": "Enter value between 0.1-1.0",
      "description": "DynamoDB read throughput ratio",
      "id": "myDDBReadThroughputRatio",
      "type": "Double"
    }
  ],
  "values": {
    "myDDBReadThroughputRatio": "0.25"
  }
}
EOF
}
`

const datapipelineConfigUpdateDef = `
provider "datapipeline" {
  region  = "us-west-2"
  profile = "twitch-feed-dev"
}

resource datapipeline "testing" {
  name      = "testing-pipeline-unittest"

  #  description = "this is testing terraform"
  definition = <<EOF
{
  "objects": [{
      "period": "8 Days",
      "name": "Every 8 days b",
      "id": "DefaultSchedule",
      "type": "Schedule",
      "startAt": "FIRST_ACTIVATION_DATE_TIME"
    }],
  "parameters": [
    {
      "default": "0.25",
      "watermark": "Enter value between 0.1-1.0",
      "description": "DynamoDB read throughput ratio",
      "id": "myDDBReadThroughputRatio",
      "type": "Double"
    }
  ],
  "values": {
    "myDDBReadThroughputRatio": "0.25"
  }
}
EOF
}
`
