package main

import (
	"code.justin.tv/feeds/clients"
	. "code.justin.tv/feeds/verify-migration/cmd"
	"fmt"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
	"net/http"
	"sort"
	"strconv"
	"time"
)

const (
	feedsEdgeHost = "http://feeds-edge.production.us-west2.justin.tv"
	audreyHost    = "http://audrey.prod.us-west2.justin.tv:9090"
)

var numPosts = 0
var numComments = 0

type Emote struct {
	Id    int `json:"id"`
	Start int `json:"start"`
	End   int `json:"end"`
	Set   int `json:"set"`
}

type Embed struct {
	RequestURL string `json:"request_url"`
}

type AudreyFeed struct {
	Posts []AudreyPost `json:"posts"`
}

type AudreyPost struct {
	ID        string         `json:"id"`
	UserID    int64          `json:"user_id"`
	Body      string         `json:"body"`
	CreatedAt time.Time      `json:"created_at"`
	Deleted   bool           `json:"deleted"`
	Emotes    []Emote        `json:"emotes"`
	Embeds    []Embed        `json:"embeds"`
	Comments  AudreyComments `json:"comments"`
}

type EdgeFeed struct {
	Posts []EdgePost `json:"posts"`
}

type EdgePost struct {
	ID        string       `json:"id"`
	UserID    string       `json:"user_id"`
	Body      string       `json:"body"`
	CreatedAt time.Time    `json:"created_at"`
	Deleted   bool         `json:"deleted"`
	Emotes    []Emote      `json:"emotes"`
	Embeds    []Embed      `json:"embeds"`
	Comments  EdgeComments `json:"comments"`
}

type AudreyComments struct {
	Comments []AudreyComment `json:"comments"`
}

type AudreyComment struct {
	ID        string    `json:"id"`
	UserID    int64     `json:"user_id"`
	Body      string    `json:"body"`
	CreatedAt time.Time `json:"created_at"`
	Deleted   bool      `json:"deleted"`
}

type EdgeComments struct {
	Comments []EdgeComment `json:"items"`
}

type EdgeComment struct {
	ID        string    `json:"id"`
	UserID    string    `json:"user_id"`
	Body      string    `json:"body"`
	CreatedAt time.Time `json:"created_at"`
	Deleted   bool      `json:"deleted"`
}

func getFeedFromFeedsEdge(ctx context.Context, client *http.Client, userID string) (*EdgeFeed, error) {
	path := fmt.Sprintf("/v1/feeds/c:%s", userID)
	var feed EdgeFeed
	err := clients.DoHTTP(ctx, client, "GET", feedsEdgeHost, path, nil, nil, &feed)
	if err != nil {
		return nil, err
	}
	return &feed, nil
}

func getFeedFromAudrey(ctx context.Context, client *http.Client, userID string) (*AudreyFeed, error) {
	path := fmt.Sprintf("/v1/feed/%s/posts", userID)
	var feed AudreyFeed
	err := clients.DoHTTP(ctx, client, "GET", audreyHost, path, nil, nil, &feed)
	if err != nil {
		return nil, err
	}
	return &feed, nil
}

type ByStart []Emote

func (a ByStart) Len() int {
	return len(a)
}

func (a ByStart) Swap(i, j int) {
	a[i], a[j] = a[j], a[i]
}

func (a ByStart) Less(i, j int) bool {
	return a[i].Start < a[j].Start
}

func main() {
	ctx := context.Background()
	var client http.Client

	userIDs := ReadIDs("users.txt")
	Randomize(userIDs)

	for i, userID := range userIDs {
		var t bool
		h := "feed:" + userID
		a, err := getFeedFromAudrey(ctx, &client, userID)
		t = Test(t, h+":audrey:err", ShouldBeNil, err)
		t = Test(t, h+":audrey:feed", ShouldNotBeNil, a)

		e, err := getFeedFromFeedsEdge(ctx, &client, userID)
		t = Test(t, h+":edge:err", ShouldBeNil, err)
		t = Test(t, h+":edge:feed", ShouldNotBeNil, e)

		t = checkPosts(t, h, e.Posts, a.Posts)
		if i%10 == 0 {
			fmt.Printf("Checked %d feeds, %d posts, %d comments\n", i, numPosts, numComments)
		}
		if t {
			return
		}
	}
}

func checkPosts(t bool, h string, actual []EdgePost, expected []AudreyPost) bool {
	h = h + ":posts"
	if !t && len(expected) == 10 && len(actual) < len(expected) {
		// Sometimes in edge we see less posts because they are deleted.
	} else {
		t = Test(t, h+":len", ShouldEqual, len(actual), len(expected))
	}
	if t {
		return t
	}
	for i := 0; i < len(actual); i++ {
		numPosts++
		h := h + ":" + strconv.Itoa(i)
		ap := actual[i]
		ep := expected[i]
		t = Test(t, h+":user_id", ShouldEqual, ap.UserID, fmt.Sprint(ep.UserID))
		t = Test(t, h+":body", ShouldEqual, ap.Body, ep.Body)
		t = Test(t, h+":created_at", ShouldEqual, ap.CreatedAt.Unix(), ep.CreatedAt.Unix())
		t = Test(t, h+":deleted", ShouldEqual, ap.Deleted, ep.Deleted)
		//t = checkEmotes(t, h, ap.Emotes, ep.Emotes)
		//t = checkEmbeds(t, h, ap.Embeds, ep.Embeds)
		t = checkComments(t, h, ap.Comments, ep.Comments)
	}
	return t
}

func checkComments(t bool, h string, actual EdgeComments, expected AudreyComments) bool {
	h = h + ":comments"
	if !t && len(expected.Comments) == 5 && len(actual.Comments) < len(expected.Comments) {
		// Sometimes in edge we see less comments because they are deleted.
	} else {
		t = Test(t, h+":len", ShouldEqual, len(actual.Comments), len(expected.Comments))
	}
	if t {
		return t
	}
	for i := 0; i < len(actual.Comments); i++ {
		numComments++
		h := h + ":" + strconv.Itoa(i)
		ac := actual.Comments[i]
		ec := expected.Comments[i]
		t = Test(t, h+":user_id", ShouldEqual, ac.UserID, fmt.Sprint(ec.UserID))
		t = Test(t, h+":body", ShouldEqual, ac.Body, ec.Body)
		t = Test(t, h+":created_at", ShouldEqual, ac.CreatedAt.Unix(), ec.CreatedAt.Unix())
		t = Test(t, h+":deleted", ShouldEqual, ac.Deleted, ec.Deleted)
	}
	return t
}

func checkEmotes(t bool, h string, actual []Emote, expected []Emote) bool {
	t = Test(t, h+":emotes", ShouldEqual, len(actual), len(expected))
	if t {
		return t
	}
	sort.Sort(ByStart(actual))
	sort.Sort(ByStart(expected))
	for i := 0; i < len(actual); i++ {
		h := h + ":emotes:" + strconv.Itoa(i)
		t = Test(t, h, ShouldResemble, actual[i], expected[i])
	}
	return t
}

func checkEmbeds(t bool, h string, actual []Embed, expected []Embed) bool {
	t = Test(t, h+":embeds", ShouldEqual, len(actual), len(expected))
	if t {
		return t
	}
	for i := 0; i < len(actual); i++ {
		h := h + ":embeds:" + strconv.Itoa(i)
		t = Test(t, h, ShouldEqual, actual[i].RequestURL, expected[i].RequestURL)
	}
	return t
}
