package main

import (
	"encoding/json"
	"fmt"
	"net/http"
	"os"
	"os/signal"
	"sync"
	"time"

	"runtime/debug"

	"strings"

	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/elb"
	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"
)

const (
	teamName    = "feeds"
	serviceName = "verify-asg"
)

var instance = service{
	osExit:  os.Exit,
	sigChan: make(chan os.Signal),
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion: CodeVersion,
	},
}

// CodeVersion is set by go build.  Should be the SHA1 of the code when it was built.
var CodeVersion string

type config struct {
}

type service struct {
	osExit func(code int)

	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner
	sigChan       chan os.Signal
	configs       struct {
		config config
	}
	checkingService checkingService
}

func (f *service) setup() error {
	if err := f.serviceCommon.Setup(); err != nil {
		return err
	}
	return nil
}

type checkingService struct {
	Elastic *elb.ELB
	Ec      *ec2.EC2
	Log     log.Logger
	OnExit  chan os.Signal
}

func (c *checkingService) Start() error {
	defer func() {
		if err := recover(); err != nil {
			c.Log.Log("PANIC on service")
			debug.PrintStack()
		}
		time.Sleep(time.Second)
		c.OnExit <- nil
	}()
	c.Log.Log("Starting checking service")
	instMap, err := instancesInELB(c.Elastic)
	c.Log.Log("numinst", len(instMap), "loaded instances")
	if err != nil {
		c.Log.Log("err", err, "unable to load instances from ELB")
		return err
	}
	badInst := false
	for lbName, instances := range instMap {
		c.Log.Log("inst", lbName, "Checking instance")
		if strings.Contains(lbName, "integration") {
			c.Log.Log("inst", lbName, "Skipping integration env")
			continue
		}
		instanceIPs, err := privateIPs(instances, c.Ec)
		if err != nil {
			c.Log.Log("inst", lbName, "err", err, "unable to load private IPs for instance")
			continue
		}
		c.Log.Log("inst", lbName, "inst_count", len(instanceIPs), "Found instances")
		codeVersions, err := fetchCodeVersions(instanceIPs, c.Log)
		if err != nil {
			c.Log.Log("inst", lbName, "err", err, "unable to load private IPs for instance")
			continue
		}
		if err := verifyCodeVersions(codeVersions, c.Log); err != nil {
			badInst = true
			c.Log.Log("inst", lbName, "err", err, "invalid code versions")
			continue
		}
	}
	if badInst {
		c.Log.Log("---------------------------------BAD-------------------------")
	} else {
		c.Log.Log("--------------------------------GOOD-------------------------")
	}
	return nil
}

func instancesInELB(elastic *elb.ELB) (map[string][]string, error) {
	ret := make(map[string][]string)
	var token *string
	for {
		in := &elb.DescribeLoadBalancersInput{
			Marker: token,
		}
		out, err := elastic.DescribeLoadBalancers(in)
		if err != nil {
			return nil, err
		}

		for _, desc := range out.LoadBalancerDescriptions {
			insts := make([]string, 0, len(desc.Instances))
			for _, inst := range desc.Instances {
				insts = append(insts, *inst.InstanceId)
			}
			ret[*desc.LoadBalancerName] = insts
		}
		token = out.NextMarker
		if token == nil {
			return ret, nil
		}
	}
}

func privateIPs(instances []string, ec *ec2.EC2) ([]string, error) {
	var token *string
	asPointers := make([]*string, 0, len(instances))
	for _, i := range instances {
		asPointers = append(asPointers, &i)
	}
	allPrivateIPs := make([]string, 0, 1024)
	for {
		in := &ec2.DescribeInstancesInput{
			NextToken:   token,
			InstanceIds: asPointers,
		}
		out, err := ec.DescribeInstances(in)
		if err != nil {
			return nil, err
		}
		for _, res := range out.Reservations {
			for _, inst := range res.Instances {
				allPrivateIPs = append(allPrivateIPs, *inst.PrivateIpAddress)
			}
		}
		token = out.NextToken
		if token == nil {
			return allPrivateIPs, nil
		}
	}
}

type codeOutput struct {
	CodeVersion string `json:"code_version"`
}

func fetchCodeVersions(ips []string, logger log.Logger) (map[string]string, error) {
	ret := make(map[string]string, len(ips))
	var mu sync.Mutex
	eg, _ := errgroup.WithContext(context.Background())
	for _, ip := range ips {
		ip := ip
		eg.Go(func() error {
			curlURL := fmt.Sprintf("http://%s:6060/debug/vars?filter=code_version", ip)
			resp, err := http.Get(curlURL)
			if err != nil {
				logger.Log("host", ip, "err", err, "Unable to connect to a host")
				return nil
			}
			var out codeOutput
			if err := json.NewDecoder(resp.Body).Decode(&out); err != nil {
				return fmt.Errorf("Invalid body for host %s: %s", ip, err)
			}
			mu.Lock()
			ret[ip] = out.CodeVersion
			mu.Unlock()
			return nil
		})
	}
	return ret, eg.Wait()
}

func verifyCodeVersions(versions map[string]string, logger log.Logger) error {
	lastVersion := ""
	lastIP := ""
	for ip, version := range versions {
		if lastVersion == "" {
			lastVersion = version
			lastIP = ip
			continue
		}
		if version != lastVersion {
			return fmt.Errorf("Invalid code version %s (%s) vs %s (%s)", lastVersion, lastIP, version, ip)
		}
	}
	logger.Log("code_version", lastVersion, "Verified all running same version")
	return nil
}

func (f *service) inject() {
	session, conf := service_common.CreateAWSSession(f.serviceCommon.ConfigCommon.Config)

	f.checkingService = checkingService{
		Log:     f.serviceCommon.Log,
		OnExit:  f.sigChan,
		Elastic: elb.New(session, conf...),
		Ec:      ec2.New(session, conf...),
	}
	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.serviceCommon, &f.checkingService,
		},
		SignalNotify: signal.Notify,
		SigChan:      f.sigChan,
	}
	// Cannot just pass f because f contains private members that I cannot nil check via reflection
	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"aws-sdk-go", "net/http"},
	}).Check(f, f.runner)
	res.MustBeEmpty(os.Stderr)
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		f.osExit(1)
		return
	}
	f.inject()

	if err := f.runner.Execute(); err != nil {
		service_common.SetupLogger.Log("err", err, "wait to end finished with an error")
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished main")
}

func main() {
	instance.main()
}
