package log

import "golang.org/x/net/context"

// CtxDimensions can propagate log dimensions inside a context object
type CtxDimensions struct {
}

// From returns the dimensions currently set inside a context object
func (c *CtxDimensions) From(ctx context.Context) []interface{} {
	if c == nil {
		return []interface{}{}
	}
	existing := ctx.Value(c)
	if existing == nil {
		return []interface{}{}
	}
	return existing.([]interface{})
}

// Append returns a new context that appends vals as dimensions for logging
func (c *CtxDimensions) Append(ctx context.Context, vals ...interface{}) context.Context {
	if c == nil {
		return ctx
	}
	if len(vals) == 0 {
		return ctx
	}
	if len(vals)%2 != 0 {
		panic("Expected even number of values")
	}
	existing := c.From(ctx)
	newArr := make([]interface{}, 0, len(existing)+len(vals))
	newArr = append(newArr, existing...)
	newArr = append(newArr, vals...)
	return context.WithValue(ctx, c, newArr)
}

// With returns a new Context object that appends the dimensions inside ctx with the context logCtx
func (c *CtxDimensions) With(ctx context.Context, log Logger) *Context {
	return NewContext(log).With(c.From(ctx)...)
}

// ContextLogger is a logger that can fetch optional dimensions from a logged context
type ContextLogger struct {
	Dims *CtxDimensions
	Logger Logger
}

// LogCtx does a log with added context dimensions
func (c *ContextLogger) LogCtx(ctx context.Context, keyvals ...interface{}) {
	prevDims := c.Dims.From(ctx)
	newArr := make([]interface{}, 0, len(prevDims)+len(keyvals))
	newArr = append(newArr, prevDims...)
	newArr = append(newArr, keyvals...)
	c.Logger.Log(newArr...)
}

func (c *ContextLogger) Disabled() bool {
	return c == nil || IsDisabled(c.Logger)
}

// Log looks for a "ctx" key with a context and calls LogCtx with it
func (c *ContextLogger) Log(keyvals ... interface{}) {
	ctx, keyvals2 := findContext(keyvals...)
	if ctx == nil {
		c.Logger.Log(keyvals2...)
		return
	}
	c.LogCtx(ctx, keyvals2...)
}

func (c *ContextLogger) LoggerContext(ctx context.Context) *Context {
	return NewContext(c.Logger).With(c.Dims.From(ctx)...)
}

func findContext(keyvals ...interface{}) (context.Context, []interface{}) {
	for k :=0;k<len(keyvals);k +=2 {
		if k == len(keyvals) - 1 {
			continue
		}
		sval, ok := keyvals[k].(string)
		if ok && sval == "ctx" {
			nexval, ok := keyvals[k+1].(context.Context)
			if !ok {
				continue
			}
			otherKeyvals := make([]interface{}, 0, len(keyvals) - 2)
			otherKeyvals = append(otherKeyvals, keyvals[0:k-1]...)
			otherKeyvals = append(otherKeyvals, keyvals[k+2:]...)
			return nexval, otherKeyvals
		}
	}
	return nil, keyvals
}

// ElevatedLog will move debug logs into normal logs if they have a context with a Value of
// ElevateKey.  Unlike other parts of the log pacakge, it makes a somewhat opinionated view that the only log levels
// that matter are debugging and normal logging.
type ElevatedLog struct {
	NormalLog ContextLogger
	DebugLog  ContextLogger
	ElevateKey interface{}
}

func (e *ElevatedLog) Disabled() bool {
	return e == nil || (IsDisabled(&e.NormalLog) && IsDisabled(&e.DebugLog))
}

// Log just does the normal logger
func (e *ElevatedLog) Log(vals ...interface{}) {
	if e != nil {
		e.NormalLog.Log(vals...)
	}
}

// LogCtx does a normal log with this context
func (e *ElevatedLog) LogCtx(ctx context.Context, vals ...interface{}) {
	if e != nil {
		e.NormalLog.LogCtx(ctx, vals...)
	}
}

// Debug does a debug logger log, but elevates it if it has a context with ElevateKey
func (e *ElevatedLog) Debug(vals ...interface{}) {
	if e == nil {
		return
	}
	ctx, keyvals := findContext(vals...)
	if ctx == nil {
		e.DebugLog.Log(keyvals...)
		return
	}
	e.DebugCtx(ctx, keyvals...)
}

func (e *ElevatedLog) DebugLogForContext(ctx context.Context) ContextLogger {
	if e == nil {
		return ContextLogger{Logger: Discard}
	}
	if e.isElevated(ctx) {
		return e.NormalLog
	}
	return e.DebugLog
}

func (e *ElevatedLog) isElevated(ctx context.Context) bool {
	if e.ElevateKey != nil {
		exists := ctx.Value(e.ElevateKey)
		return exists != nil
	}
	return false
}

// DebugCtx does a debug log, unless the context has ElevateKey, in which case it uses normal logger
func (e *ElevatedLog) DebugCtx(ctx context.Context, vals ...interface{}) {
	xy := e.DebugLogForContext(ctx)
	xy.LogCtx(ctx, vals...)
}
