package rollbar

import (
	"bytes"
	"encoding/json"
	"fmt"
	"hash/adler32"
	"hash/crc32"
	"io"
	"io/ioutil"
	"net/http"
	"os"
	"reflect"
	"runtime"
	"strings"
	"time"

	"golang.org/x/net/context"
)

// DefaultConfiguration is used when no configuration is specified.  It's usually OK to ignore this, but it's public
// if you really want to change it.
var DefaultConfiguration = DataOptionals{
	Platform: runtime.GOOS,
	Language: "go",
	Notifier: Notifier{
		Name:    LibraryName,
		Version: Version,
	},
}

func init() {
	hostname, err := os.Hostname()
	if err != nil {
		hostname = "unknown<err>"
	}
	DefaultConfiguration.Server.Host = hostname
	DefaultConfiguration.CodeVersion = CodeVersion
}

// CodeVersion can use the linker to set this variable (if desired)
var CodeVersion string

const (
	// LibraryName is the name of this library: sent with the request
	LibraryName = "rollbar2"
	// Version of this library: sent with the request
	Version = "0.0.1"
	// DefaultEndpoint is where the library sends data by default
	DefaultEndpoint = "https://api.rollbar.com/api/1/item/"
	// DefaultDrainSize is the default HTTP connection draining size
	DefaultDrainSize = 1024
	// DefaultEnvironment is sent if no env is specified in the client
	DefaultEnvironment = "development"
)

// Client connects to and sends rollbar values
type Client struct {
	// AccessToken is required and is used to authenticate the request
	AccessToken string
	// Environment is required and defautls to DefaultEnvironment
	Environment string

	// Enpoint is optional and defaults to rollbar
	Endpoint string
	// HTTPClient is optional and used to send the request
	HTTPClient http.Client
	// MessageDefaults are optional and allow you to specify default values with the request.  If you change
	// this you probably want to merge it with DefaultConfiguration
	MessageDefaults *DataOptionals
	// DrainSize is optional and used to drain the HTTP connection after it is sent
	DrainSize int64
	// Now is optional and used to get the current time
	Now func() time.Time
}

// Response is the result of a message send
type Response struct {
	Err    int            `json:"err"`
	Result ResponseResult `json:"result"`
}

// ResponseResult is part of rollbar's response object.  See rollbar documentation for what it means.
type ResponseResult struct {
	UUID string `json:"uuid"`
}

func (c *Client) url() string {
	if c.Endpoint != "" {
		return c.Endpoint
	}
	return DefaultEndpoint
}

func (c *Client) now() time.Time {
	if c.Now == nil {
		return time.Now()
	}
	return c.Now()
}

func (c *Client) maxDrainSize() int64 {
	if c.DrainSize != 0 {
		return c.DrainSize
	}
	return DefaultDrainSize
}

func (c *Client) environment() string {
	if c.Environment != "" {
		return c.Environment
	}
	return DefaultEnvironment
}

// MessageBody returns a request body that can be used for message requests.  Send it with the Send method.
func MessageBody(level Level, msg string) *Data {
	return &Data {
		Body: &Body{
			Message: &Message{
				Body: msg,
			},
		},
		DataOptionals: DataOptionals{
			Level: level,
		},
	}
}

// TraceBody returns a message body that can be used for Trace requests.  Send it with the Send method.
func TraceBody(level Level, err error) *Data {
	d := &Data {
		Body: &Body{
			Trace: &Trace{
				Frames: buildStack(1, err),
				Exception: Exception{
					Class:   errorClass(err),
					Message: err.Error(),
				},
			},
		},
		DataOptionals: DataOptionals{
			Level: level,
		},
	}
	d.Fingerprint = calculateFingerprint(d.Body.Trace.Frames, err, true)
	return d
}

// Message sends a message to rollbar.
func (c *Client) Message(ctx context.Context, level Level, msg string) (*Response, error) {
	d := MessageBody(level, msg)
	return c.Send(ctx, d)
}

// Trace sends an error to Rollbar.  If the error object implements StackTrace() []uintptr then that stack trace will be
// used instead.
func (c *Client) Trace(ctx context.Context, level Level, err error) (*Response, error) {
	d := TraceBody(level, err)
	return c.Send(ctx, d)
}

func (c *Client) setupRequest(ctx context.Context, d *Data) (*http.Response, error) {
	d.MergeFrom(c.MessageDefaults)
	d.Timestamp = c.now().Unix()
	toSend := itemToSend{
		AccessToken: c.AccessToken,
		Data: sendData{
			Environment: c.environment(),
			Data: d,
		},
	}
	jsonBody := &bytes.Buffer{}
	if err := json.NewEncoder(jsonBody).Encode(toSend); err != nil {
		return nil, err
	}
	req, err := http.NewRequest("POST", c.url(), jsonBody)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Content-Type", "application/json")
	req.Cancel = ctx.Done()
	resp, err := c.HTTPClient.Do(req)
	if err != nil {
		return nil, err
	}
	return resp, nil
}

// Send a constructed Body.  You may want to use Trace or Message as they are usually easier to interact with.
func (c *Client) Send(ctx context.Context, d *Data) (sendResponse *Response, retErr error) {
	var resp *http.Response
	var err error
	if resp, err = c.setupRequest(ctx, d); err != nil {
		return nil, err
	}
	defer func() {
		_, e2 := io.CopyN(ioutil.Discard, resp.Body, c.maxDrainSize())
		if e2 != nil && e2 != io.EOF && retErr == nil {
			retErr = e2
		}
		e3 := resp.Body.Close()
		if e3 != nil && retErr == nil {
			retErr = e3
		}
	}()
	if resp.StatusCode != http.StatusOK {
		badBody := bytes.Buffer{}
		_, err := io.CopyN(&badBody, resp.Body, c.maxDrainSize())
		return nil, &responseErr{
			body:     badBody,
			httpcode: resp.StatusCode,
			wrapped:  err,
		}
	}
	sendResponse = &Response{}
	if err := json.NewDecoder(resp.Body).Decode(sendResponse); err != nil {
		return nil, err
	}
	if sendResponse.Err != 0 {
		return sendResponse, &responseErr{
			responseCode: sendResponse.Err,
		}
	}
	return sendResponse, nil
}

func cause(err error) error {
	type causer interface {
		Cause() error
		error
	}
	if cerr, ok := err.(causer); ok {
		return cerr
	}
	return err
}

func errorClass(err error) string {
	err = cause(err)
	class := reflect.TypeOf(err).String()
	if class == "" {
		return "panic"
	} else if class == "*errors.errorString" {
		checksum := adler32.Checksum([]byte(err.Error()))
		return fmt.Sprintf("{%x}", checksum)
	} else {
		return strings.TrimPrefix(class, "*")
	}
}

func StackFrameFingerprint(frames []Frame) string {
	return calculateFingerprint(frames, nil, false)
}

func calculateFingerprint(frames []Frame, err error, useErrString bool) string {
	err = cause(err)
	hash := crc32.NewIEEE()

	// If the Error() string contains random junk, we can signal inside the error what to hash by.  Note we still
	// use the stack trace, so sometimes returning "" is enough
	type errFingerprint interface {
		Fingerprint() string
	}
	if fp, ok := err.(errFingerprint); ok {
		fmt.Fprint(hash, fp.Fingerprint())
	} else if useErrString {
		fmt.Fprint(hash, err.Error())
	}
	for _, frame := range frames {
		fmt.Fprintf(hash, "%s%s%d", frame.Filename, frame.Method, frame.Line)
	}
	return fmt.Sprintf("%x", hash.Sum32())

}
