package service_common

import (
	"math/rand"
	"strings"
	"sync/atomic"
	"time"

	"code.justin.tv/feeds/distconf"
)

// MultiErr is a struct that implements the error interface for consolidating multiple errors
type MultiErr struct {
	errs []error
}

func (e *MultiErr) Error() string {
	r := make([]string, 0, len(e.errs))
	for _, err := range e.errs {
		r = append(r, err.Error())
	}
	return strings.Join(r, " | ")
}

var _ error = &MultiErr{}

// ConsolidateErrors merges a slice of errors to a single MultiErr error
func ConsolidateErrors(errs []error) error {
	retErrs := make([]error, 0, len(errs))
	for _, err := range errs {
		if err != nil {
			if multiErr, ok := err.(*MultiErr); ok {
				retErrs = append(retErrs, multiErr.errs...)
			} else {
				retErrs = append(retErrs, err)
			}
		}
	}
	if len(retErrs) == 0 {
		return nil
	}
	if len(retErrs) == 1 {
		return retErrs[0]
	}
	return &MultiErr{
		errs: retErrs,
	}
}

// ExponentialBackoff sleeps in backoffs if Check() gets a non nil error, and resets the sleep delay if Check(nil)
type ExponentialBackoff struct {
	SleepIncr  *distconf.Duration
	Multiplier *distconf.Float
	MaxSleep   *distconf.Duration
	Sleep      func(time.Duration)

	currentDelay int64
}

func (e *ExponentialBackoff) sleep(t time.Duration) {
	t = time.Duration(int64(rand.Float64()*float64(t.Nanoseconds())) + 1)
	if e.Sleep == nil {
		time.Sleep(t)
		return
	}
	e.Sleep(t)
}

// Check an error, sleeping if err != nil (the second time)
func (e *ExponentialBackoff) Check(err error) {
	if err == nil {
		atomic.StoreInt64(&e.currentDelay, 0)
		return
	}
	toSleep := atomic.LoadInt64(&e.currentDelay)
	if toSleep != 0 {
		e.sleep(time.Duration(toSleep))
	}
	toSleep += int64(float64(e.SleepIncr.Get().Nanoseconds())*e.Multiplier.Get() + 1)
	if toSleep > e.MaxSleep.Get().Nanoseconds() {
		toSleep = e.MaxSleep.Get().Nanoseconds()
	}
	atomic.StoreInt64(&e.currentDelay, toSleep)
}
