package service_common

import (
	"encoding/json"
	"io"
	"net/http"
	"strconv"
	"time"

	"net"
	"strings"

	"code.justin.tv/feeds/log"
	"golang.org/x/net/context"
)

// HTTPStatusOk returns a simple OK message. Useful for health check msgs
type HTTPStatusOk struct {
	Log     log.ContextLogger
	Message string
}

func (h *HTTPStatusOk) message() string {
	if h.Message == "" {
		return "OK"
	}
	return h.Message
}

// ServeHTTP writes out OK
func (h *HTTPStatusOk) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	w.Header().Add("Cache-Control", "no-cache, no-store, must-revalidate")
	_, err := io.WriteString(w, h.message())
	if err != nil && h.Log.Logger != nil {
		h.Log.LogCtx(r.Context(), "err", err, "health check failed to respond")
	}
}

func httpCodeToClass(statusCode int) string {
	for i := 200; i < 900; i += 100 {
		if statusCode >= i && statusCode < i+100 {
			return strconv.Itoa(i/100) + "XX"
		}
	}
	return "?XX"
}

// ErrorWithCode is an error type that specifies its HTTP status code
type ErrorWithCode interface {
	HTTPCode() int
}

// CodedError implements ErrorWithCode wrapping an error
type CodedError struct {
	Code int
	Err  error
}

func (c *CodedError) Error() string {
	return c.Err.Error()
}

// Cause returns the wrapped error
func (c *CodedError) Cause() error {
	return c.Err
}

// HTTPCode returns the wrapped http code
func (c *CodedError) HTTPCode() int {
	return c.Code
}

// UserErrorMsg is a error code that has its own custom message
type UserErrorMsg interface {
	UserMessage() string
}

type responseError struct {
	Message string `json:"message"`
	ID      string `json:"id,omitempty"`
}

// JSONHandler has common handling code for HTTP functions that return JSON objects
type JSONHandler struct {
	Stats        *StatSender
	Log          *log.ElevatedLog
	ItemCallback func(ctx context.Context, req *http.Request) (interface{}, error)
}

func (t *JSONHandler) loggerForResponse(item interface{}, err error, code int) log.ContextLogger {
	if code >= 500 && code <= 599 {
		return t.Log.NormalLog
	}
	return t.Log.DebugLog
}

// RetriedListen continues to attempt to listen on a port if the port is taken, until the port is free.  This is important
// because our deploy process restarts a service a bit too quickly during deploy and may think it just is a failed push
func RetriedListen(ctx context.Context, addr string, retryTime time.Duration, sleepTime time.Duration) (net.Listener, error) {
	endTime := time.Now().Add(retryTime)
	var listener net.Listener
	var lastError error
	for {
		listener, lastError = net.Listen("tcp", addr)
		if lastError == nil {
			return listener, nil
		}
		if endTime.Before(time.Now()) {
			return nil, lastError
		}
		if strings.Contains(lastError.Error(), "address already in use") {
			select {
			case <-time.After(sleepTime):
			case <-ctx.Done():
				return nil, ctx.Err()
			}
			continue
		}
		return nil, lastError
	}
}

// ServeHTTPC serves out a JSON object using ItemCallback
func (t *JSONHandler) ServeHTTPC(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	code := http.StatusOK
	var i interface{}
	var callbackErr error
	defer func(startTime time.Time) {
		endTime := time.Now()
		totalTime := endTime.Sub(startTime)
		t.Stats.IncC("code."+httpCodeToClass(code), 1, 1)
		t.Stats.TimingDurationC("time", totalTime, 1)
		xy := t.loggerForResponse(i, callbackErr, code)
		xy.LogCtx(req.Context(), "url", req.URL, "code", code, "err", callbackErr, "time", totalTime, "Response completed")
	}(time.Now())
	i, callbackErr = t.ItemCallback(ctx, req)
	rw.Header().Set("Content-Type", "application/json")
	if callbackErr != nil {
		if e, ok := callbackErr.(ErrorWithCode); ok {
			code = e.HTTPCode()
		} else {
			code = http.StatusInternalServerError
		}
	}
	rw.WriteHeader(code)
	if callbackErr != nil {
		re := responseError{
		// TODO: associate an ID with the log messages?
		//ID: "1",
		}
		if e, ok := callbackErr.(UserErrorMsg); ok {
			re.Message = e.UserMessage()
		} else {
			re.Message = callbackErr.Error()
		}
		if err := json.NewEncoder(rw).Encode(re); err != nil {
			t.Log.Log("err", err, "unable to encode error response back to client")
		}
		return
	}
	if err := json.NewEncoder(rw).Encode(i); err != nil {
		t.Log.Log("err", err, "unable to encode response object response back to client")
	}
}
