package service_common

import (
	"fmt"
	"io"
	"reflect"
	"strconv"
	"strings"
)

type CheckResult struct {
	res []string
}

func (c *CheckResult) MustBeEmpty(out io.Writer) {
	if len(c.res) == 0 {
		return
	}
	x := fmt.Sprintf("Found nil things.  Maybe they should be set?  If not, try nilcheck ignore or nodepth  %s", strings.Join(c.res, ","))
	fmt.Fprintf(out, x)
	panic(x)
}

type NilCheck struct {
	IgnoredPackages []string
}

// NilCheck tells you if you forgot to set some values in your struct.
func (n *NilCheck) Check(s ...interface{}) CheckResult {
	r := []string{}
	for i := range s {
		r = append(r, n.nilCheckImpl(s[i], "", true, make(map[uintptr]struct{}))...)
	}
	return CheckResult{
		res: r,
	}
}

func (n *NilCheck) isPackageIgnored(pkg string) bool {
	for _, p := range n.IgnoredPackages {
		if strings.Contains(pkg, p) {
			return true
		}
	}
	return false
}

func (n *NilCheck) nilCheckImplVal(rv reflect.Value, depth string, allowDepth bool, loopDetection map[uintptr]struct{}) []string {
	if !rv.IsValid() {
		return []string{depth}
	}
	if n.isPackageIgnored(rv.Type().PkgPath()) {
		return nil
	}
	if rv.Kind() == reflect.Ptr || rv.Kind() == reflect.Interface {
		if rv.IsNil() {
			return []string{depth}
		}
		if _, exists := loopDetection[rv.Pointer()]; exists {
			return nil
		}
		loopDetection[rv.Pointer()] = struct{}{}
		return n.nilCheckImpl(rv.Elem().Interface(), depth, allowDepth, loopDetection)
	}
	if rv.Kind() == reflect.Func {
		if rv.IsNil() {
			return []string{depth}
		}
		return nil
	}
	if rv.Kind() == reflect.Slice {
		if _, exists := loopDetection[rv.Pointer()]; exists {
			return nil
		}
		loopDetection[rv.Pointer()] = struct{}{}
		ret := []string{}
		for i := 0; i < rv.Len(); i++ {
			sField := rv.Index(i)
			if !sField.CanInterface() {
				continue
			}
			ret = append(ret, n.nilCheckImpl(sField.Interface(), depth+"/"+strconv.Itoa(i), allowDepth, loopDetection)...)
		}
		return ret
	}

	if rv.Kind() != reflect.Struct && rv.Kind() != reflect.Interface {
		return nil
	}
	if !allowDepth {
		return nil
	}

	ret := []string{}
	for i := 0; i < rv.NumField(); i++ {
		if strings.Contains(rv.Type().Field(i).Tag.Get("nilcheck"), "ignore") {
			continue
		}
		noAllowDepth := false
		if strings.Contains(rv.Type().Field(i).Tag.Get("nilcheck"), "nodepth") {
			noAllowDepth = true
		}
		sField := rv.Field(i)
		if n.isPackageIgnored(rv.Type().Field(i).PkgPath) {
			continue
		}
		fieldName := rv.Type().Field(i).Name

		if !sField.CanInterface() {
			continue
		}
		ret = append(ret, n.nilCheckImpl(sField.Interface(), depth+"/"+fieldName, allowDepth && !noAllowDepth, loopDetection)...)
	}
	return ret
}

func (n *NilCheck) nilCheckImpl(s interface{}, depth string, allowDepth bool, loopDetection map[uintptr]struct{}) []string {
	return n.nilCheckImplVal(reflect.ValueOf(s), depth, allowDepth, loopDetection)
}
