package service_common

import (
	"io"
	"os"
	"os/signal"
	"sync"
	"syscall"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
)

// Service is any struct that should be started
type Service interface {
	Start() error
}

// Settupable is an optional Service interface that is called before Start()
type Settupable interface {
	Setup() error
}

// ServiceRunner controls starting and running services until a SIGTERM
type ServiceRunner struct {
	Services     []Service
	Log          log.Logger
	SigChan      chan os.Signal
	SignalNotify func(c chan<- os.Signal, sig ...os.Signal)
}

func (r *ServiceRunner) runServices(wg *sync.WaitGroup) {
	wg.Add(len(r.Services))
	for _, s := range r.Services {
		go func(s Service) {
			defer func() {
				defer wg.Done()
				if rec := recover(); rec != nil {
					// Just absorb panic and log it.  Use errors.New() to get the stack trace
					err := errors.New("service paniced!  Rut row!!!")
					r.Log.Log("rec", rec, "service", s, "err", err)
				}
			}()
			r.Log.Log("service", s, "Starting service")
			err := s.Start()
			r.Log.Log("err", err, "Service finished")
		}(s)
	}
}

// Execute starts the service runner.  Runs until SIGTERM or SIGINT
func (r *ServiceRunner) Execute() error {
	r.Log.Log("service_count", len(r.Services), "Starting service execution")
	defer r.Log.Log("Service execution finished")
	if r.SigChan == nil {
		r.SigChan = make(chan os.Signal)
	}
	sigNotify := r.SignalNotify
	if sigNotify == nil {
		sigNotify = signal.Notify
	}
	for _, s := range r.Services {
		if setter, ok := s.(Settupable); ok {
			if err := setter.Setup(); err != nil {
				return err
			}
		}
	}
	var wg sync.WaitGroup
	r.runServices(&wg)

	r.Log.Log("Waiting on signal channel")
	sigNotify(r.SigChan, syscall.SIGTERM, syscall.SIGINT)
	sig := <-r.SigChan
	r.Log.Log("sig", sig, "Received signal")
	errs := make([]error, 0, len(r.Services))
	for i := len(r.Services) - 1; i >= 0; i-- {
		if closer, ok := r.Services[i].(io.Closer); ok {
			err := closer.Close()
			if err != nil {
				r.Log.Log("err", err, "service", r.Services[i], "Unable to close service")
				errs = append(errs, err)
			}
		}
	}
	r.Log.Log("Waiting on all services to finish")
	wg.Wait()
	r.Log.Log("Waiting done")
	return ConsolidateErrors(errs)
}
