package service_common

import (
	"runtime"
	"time"

	"expvar"
	"sync/atomic"

	"fmt"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"github.com/cactus/go-statsd-client/statsd"
)

// Gometrics collects common go runtime stats
type Gometrics struct {
	Stats     *StatSender
	StartTime time.Time

	lastRun     runtime.MemStats
	lastCgoCall int64
}

// Start sends common go stats until whenToStop is closed
func (g *Gometrics) Start(whenToStop chan struct{}, waitTime *distconf.Duration) {
	for {
		select {
		case <-whenToStop:
			return
		case <-time.After(waitTime.Get()):
			g.Tick()
		}
	}
}

func (g *Gometrics) Format(f fmt.State, c rune) {
	fmt.Fprint(f, "gometrics service")
}

// Tick registers the stats in statsd
func (g *Gometrics) Tick() {
	mstat := runtime.MemStats{}
	runtime.ReadMemStats(&mstat)
	now := time.Now()
	numCgoCall := runtime.NumCgoCall()

	g.Stats.GaugeC("num_goroutines", int64(runtime.NumGoroutine()), 1.0)
	g.Stats.GaugeC("maxprocs", int64(runtime.GOMAXPROCS(0)), 1.0)
	g.Stats.GaugeC("num_cpu", int64(runtime.NumCPU()), 1.0)
	g.Stats.GaugeC("process.uptime.ns", now.Sub(g.StartTime).Nanoseconds(), 1.0)
	g.Stats.IncC("num_cgo_call", numCgoCall-g.lastCgoCall, 1.0)
	g.Stats.GaugeC("mem.heap_objects", int64(mstat.HeapObjects), 1.0)

	g.Stats.IncC("mem.mallocs", int64(mstat.Mallocs-g.lastRun.Mallocs), 1.0)
	g.Stats.IncC("mem.frees", int64(mstat.Frees-g.lastRun.Frees), 1.0)
	g.Stats.IncC("mem.lookups", int64(mstat.Lookups-g.lastRun.Lookups), 1.0)
	g.Stats.IncC("mem.total_alloc", int64(mstat.TotalAlloc-g.lastRun.TotalAlloc), 1.0)
	g.Stats.IncC("gc.total_duration", int64(mstat.PauseTotalNs-g.lastRun.PauseTotalNs), 1.0)
	g.Stats.GaugeC("gc.percent_cpu", fractionToPercent(mstat.GCCPUFraction), 1.0)
	g.Stats.IncC("gc.collections", int64(mstat.NumGC-g.lastRun.NumGC), 1.0)

	g.Stats.GaugeC("mem.alloc", int64(mstat.Alloc), 1.0)
	g.Stats.GaugeC("mem.sys", int64(mstat.Sys), 1.0)
	g.Stats.GaugeC("mem.heap_alloc", int64(mstat.HeapAlloc), 1.0)
	g.Stats.GaugeC("mem.heap_sys", int64(mstat.HeapSys), 1.0)
	g.Stats.GaugeC("mem.heap_idle", int64(mstat.HeapIdle), 1.0)
	g.Stats.GaugeC("mem.heap_inuse", int64(mstat.HeapInuse), 1.0)
	g.Stats.GaugeC("mem.heap_released", int64(mstat.HeapReleased), 1.0)
	g.Stats.GaugeC("mem.stack_inuse", int64(mstat.StackInuse), 1.0)
	g.Stats.GaugeC("mem.stack_sys", int64(mstat.StackSys), 1.0)
	g.Stats.GaugeC("mem.mspan_inuse", int64(mstat.MSpanInuse), 1.0)
	g.Stats.GaugeC("mem.mspan_sys", int64(mstat.MSpanSys), 1.0)
	g.Stats.GaugeC("mem.mcache_inuse", int64(mstat.MCacheInuse), 1.0)
	g.Stats.GaugeC("mem.mcache_sys", int64(mstat.MCacheSys), 1.0)
	g.Stats.GaugeC("mem.buck_hash_sys", int64(mstat.BuckHashSys), 1.0)

	g.Stats.GaugeC("gc.gc_sys", int64(mstat.GCSys), 1.0)
	g.Stats.GaugeC("gc.other_sys", int64(mstat.OtherSys), 1.0)
	g.Stats.GaugeC("gc.next_gc", int64(mstat.NextGC), 1.0)
	g.Stats.GaugeC("gc.last_gc", int64(mstat.LastGC), 1.0)
	g.Stats.GaugeC("gc.num_gc", int64(mstat.NumGC), 1.0)

	g.lastRun = mstat
	g.lastCgoCall = numCgoCall
}

func fractionToPercent(fraction float64) int64 {
	return int64((fraction * 100) + 0.5)
}

// ErrorTracker is a catch all atomic integer for errors not worth logging
type ErrorTracker struct {
	Log         *log.ElevatedLog
	totalErrors int64
}

func (s *ErrorTracker) track(err error) {
	if err != nil {
		if new := atomic.AddInt64(&s.totalErrors, 1); new == 1 {
			s.Log.Log("err", err, "encountered first non tracked error")
		}
		s.Log.Debug("err", err, "non tracked error")
	}
}

// Var exposes the current # of total Errors seen
func (s *ErrorTracker) Var() expvar.Var {
	return expvar.Func(func() interface{} {
		return atomic.LoadInt64(&s.totalErrors)
	})
}

// NopStatSender creates a stat sender that goes nowhere
func NopStatSender() *StatSender {
	return &StatSender{
		&statsd.NoopClient{},
		&ErrorTracker{},
	}
}

// StatSender has non error returning functions that make it easier to use than a raw StatSender
type StatSender struct {
	statsd.SubStatter
	*ErrorTracker
}

// IncC is an error checked Inc
func (st *StatSender) IncC(s string, i int64, f float32) {
	st.track(st.SubStatter.Inc(s, i, f))
}

// DecC is an error checked Dec
func (st *StatSender) DecC(s string, i int64, f float32) {
	st.track(st.SubStatter.Dec(s, i, f))
}

// GaugeC is an error checked Gauge
func (st *StatSender) GaugeC(s string, i int64, f float32) {
	st.track(st.SubStatter.Gauge(s, i, f))
}

// GaugeDeltaC is an error checked GaugeDelta
func (st *StatSender) GaugeDeltaC(s string, i int64, f float32) {
	st.track(st.SubStatter.GaugeDelta(s, i, f))
}

// TimingC is an error checked Timing
func (st *StatSender) TimingC(s string, i int64, f float32) {
	st.track(st.SubStatter.Timing(s, i, f))
}

// TimingDurationC is an error checked TimingDuration
func (st *StatSender) TimingDurationC(s string, t time.Duration, f float32) {
	st.track(st.SubStatter.TimingDuration(s, t, f))
}

// SetC is an error checked Set
func (st *StatSender) SetC(s string, v string, f float32) {
	st.track(st.SubStatter.Set(s, v, f))
}

// SetIntC is an error checked SetInt
func (st *StatSender) SetIntC(s string, i int64, f float32) {
	st.track(st.SubStatter.SetInt(s, i, f))
}

// RawC is an error checked Raw
func (st *StatSender) RawC(s string, v string, f float32) {
	st.track(st.SubStatter.Raw(s, v, f))
}

// NewSubStatSender mirrors statsd NewSubStatter
func (st *StatSender) NewSubStatSender(substat string) *StatSender {
	return &StatSender{
		SubStatter:   st.SubStatter.NewSubStatter(substat),
		ErrorTracker: st.ErrorTracker,
	}
}

// NewSubStatter forwards to NewSubStatSender
func (st *StatSender) NewSubStatter(substat string) statsd.SubStatter {
	return st.NewSubStatSender(substat)
}
