package manager

import (
	"code.justin.tv/common/ddbmetrics"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

// Get will fetch and decrypt a secret, returning a pointer to a
// Secret struct.
func (m *Manager) Get(secretName string) (*Secret, error) {
	secret, err := m.get(secretName)
	if err != nil {
		return nil, err
	}
	if secret == nil {
		return nil, nil
	}
	err = m.Decrypt(secret)
	if err != nil {
		return nil, err
	}
	return secret, nil
}

// Exist check the given secret exist in the system.
// This also performs the check to make sure that the secret can
// be unmarshalled.
func (m *Manager) Exist(secretName string) (bool, error) {
	secret, err := m.get(secretName)
	return secret != nil, err
}

// GetEncrypted returns a pointer to a Secret, but doees not actually
// perform the decryption. This is used by the JSON API.
func (m *Manager) GetEncrypted(secretName string) (*Secret, error) {
	secret, err := m.get(secretName)
	return secret, err
}

func (m *Manager) get(name string) (*Secret, error) {
	queryInput := m.inputForGet(name)
	queryOutput, err := m.DynamoDB.Query(queryInput)
	if err != nil {
		return nil, err
	}
	m.metrics.Report(ddbmetrics.Read, queryOutput.ConsumedCapacity)
	if *queryOutput.Count == 0 {
		return nil, nil
	}
	secret, err := unmarshalSecret(queryOutput.Items[0])
	if err != nil {
		return nil, err
	}

	return secret, nil
}

func (m *Manager) inputForGet(name string) *dynamodb.QueryInput {
	return &dynamodb.QueryInput{
		ConsistentRead: aws.Bool(true),
		ExpressionAttributeNames: map[string]*string{
			"#N": aws.String("name"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":name": {
				S: aws.String(name),
			},
		},
		KeyConditionExpression: aws.String("#N = :name"),
		ReturnConsumedCapacity: aws.String("INDEXES"),
		TableName:              aws.String(m.TableName()),
	}
}
