package xray

import (
	"expvar"
	"os"
	"sync/atomic"
)

var defaultConfig = Config{
	Name:        os.Getenv("XRAY_TRACING_NAME"),
	DefaultName: os.Getenv("XRAY_TRACING_DEFAULT_NAME"),
	DaemonAddr:  "127.0.0.1:2000",
	Sampling:    1.0,
}

// Config controls (currently undocumented) configuration parameters for xray
type Config struct {
	Name           string
	DefaultName    string
	DaemonAddr     string
	Sampling       float64 // To disable all tracing, set to a negative number
	ServiceVersion string
}

// Configure this xray instance with this config.  It will load default values for any config values not set.
func (x *XRay) Configure(c Config) error {
	x.configMutex.Lock()
	defer x.configMutex.Unlock()
	xrayConfig := defaultConfig
	x.storedConfig = &xrayConfig

	if c.Name != "" {
		x.storedConfig.Name = c.Name
	}

	if c.DefaultName != "" {
		x.storedConfig.DefaultName = c.DefaultName
	}

	if c.Sampling != 0.0 {
		x.storedConfig.Sampling = c.Sampling
	}

	if c.ServiceVersion != "" {
		x.storedConfig.ServiceVersion = c.ServiceVersion
	}

	if c.DaemonAddr != "" {
		x.storedConfig.DaemonAddr = c.DaemonAddr
	}

	if err := x.refreshEmitter(x.storedConfig.DaemonAddr); err != nil {
		return err
	}

	return nil
}

// PartialConfigure is similar to Configure, but only sets the parts of configure that are currently unset
func (x *XRay) PartialConfigure(c Config) error {
	x.configMutex.Lock()
	defer x.configMutex.Unlock()
	if c.Name != "" {
		x.storedConfig.Name = c.Name
	}

	if c.DefaultName != "" {
		x.storedConfig.DefaultName = c.DefaultName
	}

	if c.Sampling != 0.0 {
		x.storedConfig.Sampling = c.Sampling
	}

	if c.ServiceVersion != "" {
		x.storedConfig.ServiceVersion = c.ServiceVersion
	}

	if c.DaemonAddr != "" {
		x.storedConfig.DaemonAddr = c.DaemonAddr
		if err := x.refreshEmitter(x.storedConfig.DaemonAddr); err != nil {
			return err
		}
	}

	return nil
}

func (x *XRay) Var() expvar.Var {
	return expvar.Func(func() interface{} {
		conf := x.config()
		return map[string]interface{}{
			"config": *conf,
			"stats": xrayStats{
				TotalSegments: atomic.LoadInt64(&x.xrayStats.TotalSegments),
				EmitAttempts:  atomic.LoadInt64(&x.xrayStats.EmitAttempts),
				EmittedTotal:  atomic.LoadInt64(&x.xrayStats.EmittedTotal),
				TotalErrors:   atomic.LoadInt64(&x.xrayStats.TotalErrors),
			},
		}
	})
}
