package xray

import (
	"net/http"
	"net/http/httptest"
	"testing"

	"math"
	"sync"
	"sync/atomic"

	"github.com/stretchr/testify/assert"
)

func TestEnvironmentName(t *testing.T) {
	t.Parallel()
	x, testDaemon := makeTestDaemon(t, Config{
		Name: "EnvName",
	})
	defer testDaemon.Close(t)

	h := x.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {}))
	r, _ := http.NewRequest("GET", "http://10.0.0.1/", nil)
	w := httptest.NewRecorder()
	h.ServeHTTP(w, r)

	seg, e := testDaemon.Recv()
	assert.Nil(t, e)

	assert.Equal(t, "EnvName", seg.Name)
}

func TestEnvironmentDefaultName(t *testing.T) {
	t.Parallel()
	x, testDaemon := makeTestDaemon(t, Config{
		DefaultName: "DefaultEnvName",
	})
	defer testDaemon.Close(t)

	h := x.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {}))
	r, _ := http.NewRequest("GET", "http://10.0.0.1/", nil)
	w := httptest.NewRecorder()
	h.ServeHTTP(w, r)

	seg, e := testDaemon.Recv()
	assert.Nil(t, e)

	assert.Equal(t, "DefaultEnvName", seg.Name)
}

func TestConfigName(t *testing.T) {
	t.Parallel()
	x, testDaemon := makeTestDaemon(t, Config{
		Name:        "ConfigName",
		DefaultName: "DefaultConfigName",
	})
	defer testDaemon.Close(t)

	h := x.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {}))
	r, _ := http.NewRequest("GET", "http://10.0.0.1/", nil)
	w := httptest.NewRecorder()
	h.ServeHTTP(w, r)

	seg, e := testDaemon.Recv()
	assert.Nil(t, e)

	assert.Equal(t, "ConfigName", seg.Name)
}

func TestDefaultConfigName(t *testing.T) {
	t.Parallel()
	x, testDaemon := makeTestDaemon(t, Config{
		DefaultName: "DefaultConfigName",
	})
	defer testDaemon.Close(t)

	h := x.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {}))
	r, _ := http.NewRequest("GET", "http://10.0.0.1/", nil)
	w := httptest.NewRecorder()
	h.ServeHTTP(w, r)

	seg, e := testDaemon.Recv()
	assert.Nil(t, e)

	assert.Equal(t, "DefaultConfigName", seg.Name)
}

func TestFailingDaemonAddr(t *testing.T) {
	t.Parallel()
	x := XRay{}
	assert.Error(t, x.Configure(Config{
		DaemonAddr: "fakeHost:2017",
	}))
}

func BenchmarkFloatByLock(b *testing.B) {
	f := sync.Mutex{}
	floatVal := 1.0
	limit := float64(b.N + 1)
	threadCount := 10
	wg := sync.WaitGroup{}
	wg.Add(threadCount)
	for j := 0; j < threadCount; j++ {
		go func() {
			for i := 0; i < b.N; i++ {
				v := getFloatByLock(&f, &floatVal)
				if v > limit {
					b.Error("bad value")
				}
				if i%1000 == 0 {
					setFloatByLock(&f, &floatVal, float64(i))
				}
			}
			wg.Done()
		}()
	}
	wg.Wait()

}
func setFloatByLock(mu *sync.Mutex, f *float64, val float64) {
	mu.Lock()
	*f = val
	mu.Unlock()
}
func getFloatByLock(mu *sync.Mutex, f *float64) float64 {
	mu.Lock()
	ret := *f
	mu.Unlock()
	return ret
}

func BenchmarkFloatByAtomic(b *testing.B) {
	floatVal := uint64(1)
	limit := float64(b.N + 1)
	threadCount := 10
	wg := sync.WaitGroup{}
	wg.Add(threadCount)
	for j := 0; j < threadCount; j++ {
		go func() {
			for i := 0; i < b.N; i++ {
				v := getFloatByAtomic(&floatVal)
				if v > limit {
					b.Error("bad value")
				}
				if i%1000 == 0 {
					setFloatByAtomic(&floatVal, float64(i))
				}
			}
			wg.Done()
		}()
	}
	wg.Wait()

}
func setFloatByAtomic(f *uint64, val float64) {
	asBits := math.Float64bits(val)
	atomic.StoreUint64(f, asBits)
}

func getFloatByAtomic(f *uint64) float64 {
	asInt := atomic.LoadUint64(f)
	return math.Float64frombits(asInt)
}
