# @twitch/flexo/JenkinsSlavesEcs

This module creates an ASG backed ECS Cluster based environment for Jenkins Agents within your Tiny Bubble environment.
Additionally it sets up both a Cloudwatch Dashboard, and a Cloudwatch Log Group. 

## Before running the automation

Create an IPDEV ticket in Jira, and ping us in #developer-infra.  The info we'll need is:

1. Your AWS Account ID that your ECS cluster will run in
2. A unique label for your jobs to run in (See [here][5] for more info).
    * This label will be used in all Jenkinsfile that utilize your ECS cluster.
    * For the examples in this README, we'll be using the label `IPDEV-ECS-Production`
3. If you would like to use any `ECS Agent Templates` for your cluster
    * Templates store various default values that can be used for your jobs.  Check out the [docs][4] for more info.
4. If you decide to use Templates, please submit that information via Jira along with the label discussed above.
**NOTE: Running the automation without performing the above step will result in a deployment failure**
  
## Creating your Jenkins Slave ECS Cluster
Create your cdk stack as follows:

```typescript
import * as ec2  from '@aws-cdk/aws-ec2';
import * as flexo from '@twitch/flexo';

export class JenkinsEcsStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: cdk.StackProps) {
    super(scope, id, props);

    const vpc = new ec2.Vpc(this, 'VPC');

    new flexo.JenkinsSlavesEcs(this, 'margibsECS', {
      vpc: vpc,
      instanceType: new ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MEDIUM),
      machineImage: ecs.EcsOptimizedImage.amazonLinux2(),
      logGroupName: '/JenkinsAgents',
      dashboardName: 'JenkinsECSAgents'
    });
  }
}
```

- Run the automation to build your environment!
- In the example above, in addition to setting up the ASG that will run your ECS cluster, the CDK also creates two things to help you monitor via Cloudwatch. 
`logGroupName` is a unique name where logs will be sent. `dashboardName` will create a dashboard in Cloudwatch.

- If you decide to use templating, you can submit those configurations via Jira to IPDEV. 

## Creating your ECS Cluster Job

In your `Jenkinsfile` you can create a job that looks similar to this:

```groovy
pipeline {
   agent none

   stages {
      stage('Hello') {
         agent { 
             ecs { 
                 // This is the label that you've provided the IPDEV team
                 inheritFrom 'IPDEV-ECS-Production'
                 cpu 2048
                 memory 4096
                 // Note: logDriverOptions are optional to add.  Omit this, and no logs will be stored in CloudWatch
                 // The awslogs-group value must match the logGroupName given in the above CDK
                 // The awslogs-region must match the region your ECS cluster is deployed in
                 logDriverOptions([[name: 'awslogs-group', value: '/JenkinsAgents'], [name: 'awslogs-region', value: 'us-west-2']])
             } 
         } 
         steps {
            echo 'Hello World'
          
         }
      }
   }
}
```
For a list of available options that can be specified in a Jenkinsfile for the ECS plugin, please see the link below:

* [ECS Plugin Pipeline Syntax][3]

**NOTE:** Specifying CPU, and Memory in your Jenkinsfile is directly related to the size of the instance you've created. 
For example the configuration shown above would not work with a t2.small instance. 
The CPU, and Memory would need to be set lower 1024 for CPU and 2048 for Memory (or lower). 
For an extensive list of valid values, and instance sizes please refer to the two links below:

* [Valid CPU/Memory in ECS Tasks][1]
* [Amazon EC2 Instance Types][2]

[1]:https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-cpu-memory-error.html
[2]:https://aws.amazon.com/ec2/instance-types/
[3]:https://jenkins.io/doc/pipeline/steps/amazon-ecs/
[4]:https://plugins.jenkins.io/scalable-amazon-ecs/
[5]:https://jenkins.io/doc/book/pipeline/syntax/#agent








