import { expect as expectCDK, haveResource, haveResourceLike, SynthUtils } from '@aws-cdk/assert';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as cdk from '@aws-cdk/core';
import * as flexo from '../../lib/';

test('Create Flexo Swarm Slave AL2', () => {
    const app = new cdk.App();
    const stack = new cdk.Stack(app, "TestStack");

    const vpc = new ec2.Vpc(stack, 'Vpc');
    new flexo.JenkinsSlavesSwarmASGAmazonLinux2(stack, 'Flexo', {
        vpc,
        instanceType: ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
        swarmLabels: 'testLabel',
        secretArn: 'arn:aws:secretsmanager:us-west-2:000000:secret:jenkins-slave-credentials-XYZABC'
    });

    expectCDK(stack).to(haveResource("AWS::AutoScaling::AutoScalingGroup"));
    expectCDK(stack).to(haveResourceLike("AWS::EC2::VPCEndpoint", {
        PrivateDnsEnabled: false
    }));
    expectCDK(stack).to(haveResourceLike("AWS::Route53::HostedZone", {
        Name: `git-aws.internal.justin.tv.`
    }));
    expectCDK(stack).to(haveResourceLike("AWS::CloudWatch::Dashboard"));
    expectCDK(stack).to(haveResourceLike("AWS::Route53::RecordSet", {
        Name: `git-aws.internal.justin.tv.`,
        Type: "A",
    }));
    expectCDK(stack).to(haveResourceLike("AWS::Route53::HostedZone", {
        Name: `jenkins.xarth.tv.`
    }));
    expectCDK(stack).to(haveResourceLike("AWS::Route53::RecordSet", {
        Name: `jenkins.xarth.tv.`,
        Type: "A",
    }));
    expectCDK(stack).to(haveResourceLike("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [{
                Action: "sts:AssumeRole",
                Effect: "Allow"
            }]
        }
    }));
    expect(SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});

test('Create Flexo Swarm Slave with all options AL2', () => {
    const app = new cdk.App();
    const stack = new cdk.Stack(app, "TestStack", {
        env: {
            region: "us-west-2"
        }
    });

    const vpc = new ec2.Vpc(stack, 'Vpc');
    new flexo.JenkinsSlavesSwarmASGAmazonLinux2(stack, 'Flexo', {
        vpc,
        machineImage: new ec2.GenericLinuxImage({'us-west-2': 'ami-12345'}),
        instanceType: ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
        swarmLabels: 'testLabel',
        swarmDescription: 'desc',
        swarmExecutors: 5,
        swarmMode: "exclusive",
        swarmExtraOptions: " -extra opts",
        swarmFsRoot: "/tmp/test",
        jenkinsEnvironment: "staging",
        ssm_agent_rpm: "http://test.com",
        swarm_version: "3.3",
        swarm_client_jar: "http://testother.com",
        jar_directory: "/tmp/test2",
        secretArn: 'arn:aws:secretsmanager:us-west-2:000000:secret:jenkins-slave-credentials-XYZABC'
    });

    expectCDK(stack).to(haveResource("AWS::AutoScaling::AutoScalingGroup"));
    expectCDK(stack).to(haveResourceLike("AWS::EC2::VPCEndpoint", {
        PrivateDnsEnabled: false,
    }));
    expectCDK(stack).to(haveResourceLike("AWS::Route53::HostedZone", {
        Name: `staging.jenkins.xarth.tv.`
    }));
    expectCDK(stack).to(haveResourceLike("AWS::Route53::RecordSet", {
        Name: `staging.jenkins.xarth.tv.`,
        Type: "A",
    }));
    expect(SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});

test('Create Flexo Swarm Slave with all options Ubuntu', () => {
    const app = new cdk.App();
    const stack = new cdk.Stack(app, "TestStack", {
        env: {
            region: "us-west-2"
        }
    });

    const vpc = new ec2.Vpc(stack, 'Vpc');
    new flexo.JenkinsSlavesSwarmASGUbuntu(stack, 'Flexo', {
        vpc,
        machineImage: new ec2.GenericLinuxImage({'us-west-2': 'ami-12345'}),
        instanceType: ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
        swarmLabels: 'testLabel',
        swarmDescription: 'desc',
        swarmExecutors: 5,
        swarmMode: "exclusive",
        swarmExtraOptions: " -extra opts",
        swarmFsRoot: "/tmp/test",
        jenkinsEnvironment: "staging",
        ssm_agent_rpm: "http://test.com",
        swarm_version: "3.3",
        swarm_client_jar: "http://testother.com",
        jar_directory: "/tmp/test2",
        secretArn: 'arn:aws:secretsmanager:us-west-2:000000:secret:jenkins-slave-credentials-XYZABC'
    });

    expectCDK(stack).to(haveResource("AWS::AutoScaling::AutoScalingGroup"));
    expectCDK(stack).to(haveResourceLike("AWS::EC2::VPCEndpoint", {
        PrivateDnsEnabled: false,
    }));
    expectCDK(stack).to(haveResourceLike("AWS::Route53::HostedZone", {
        Name: `staging.jenkins.xarth.tv.`
    }));
    expectCDK(stack).to(haveResourceLike("AWS::Route53::RecordSet", {
        Name: `staging.jenkins.xarth.tv.`,
        Type: "A",
    }));
    expect(SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
