# Flexo Jenkins Shared Library

- [Intro](#intro)
- [The Libraries](#the-libraries)
  * [flexo.pipeline](#flexopipeline)
  * [setBuildStatus*](#setbuildstatus)
- [Developer Setup](#developer-setup)
  * [To spin up a local Jenkins instance with shared libraries installed:](#to-spin-up-a-local-jenkins-instance-with-shared-libraries-installed)
  * [Using the included Dockerfile](#using-the-included-dockerfile)
  * [Adding a new library and jenkins test jobs](#adding-a-new-library-and-jenkins-test-jobs)
  * [Writing your Jenkinsfile](#writing-your-jenkinsfile)
  * [Unit Testing Jenkins Shared Libraries](#unit-testing-jenkins-shared-libraries)
  * [Further Reading](#further-reading)

# Intro

This repo contains the source code for the shared Jenkins library that is available as part of the Flexo platform.

[Flexo Jenkins Shared Library Documenation][library_doc]

![Flexo](imgs/flexo.gif)

# The Libraries

## flexo.pipeline

Instantiates a new Flexo pipeline based on a `template` and allowed to run on agents that match `agentLabel`. See [the Flexo docs](https://git-aws.internal.justin.tv/pages/flexo/flexo-docs/docs/pipelines/overview.html) for more information.

## setBuildStatus*

Adds the ability to update GHE commits/PRs with build statuses so you can have your PRs look like so

![SetBuildStatus](imgs/setBuildStatus.png)

**PREREQUISITES**

* [Create credentials in Jenkins that](https://www.jenkins.io/doc/book/using/using-credentials/#adding-new-global-credentials) that are of kind `Username with password`.
  * Set the username to your GHE user
  * Set the password to your PAT [(PAT creation instructions here)](https://help.github.com/en/enterprise/2.17/user/github/authenticating-to-github/creating-a-personal-access-token-for-the-command-line).  
  * The ID should be something easily distinguishable for your org, this ID is what gets fed into `credentialsId` below

**USAGE**

```groovy
@Library('flexo-libs') _

pipeline {
    agent any
    stages {
        // All you need is the credentialsId to be passed in.
        // Sane defaults will probably be fine for most use cases
        // At most, you may want to look at the example_with_more_options stage below and add a description
        stage('minimal_example') {
            steps {
                setBuildStatusPending(
                    credentialsId: 'flexo-org-service_pat'
                )
                echo 'Hello build!'
            }
            post {
                success { 
                    setBuildStatusSuccess(
                        credentialsId: 'flexo-org-service_pat',
                        description: 'This description is optional"
                    )
                }
                failure { 
                    setBuildStatusFailure(
                        credentialsId: 'flexo-org-service_pat'
                    )
                }
            }
        }
        
        stage('example_with_more_options') {
            steps {
                setBuildStatusPending(
                    description: 'Add a description for when you hover over the status in GHE',
                    context: 'By default, the context shown in GHE will be the STAGE_NAME - example_with_more_options - so we can override that here'
                    credentialsId: 'flexo-org-service_pat',
                    debug: true // For some extra debug output of what the plugin is doing (some output will be in /var/log/jenkins.log)
                )
                echo 'Hello build!'
            }
            post {
                success { 
                    setBuildStatusSuccess(
                        description: 'example success',
                        context: 'Successful Example Step',
                        credentialsId: 'flexo-org-service_pat'
                    )
                }
                failure { 
                    setBuildStatusFailure(
                        description: 'example failure',
                        credentialsId: 'flexo-org-service_pat'
                    )
                }
            }
        }
    }
}

```

---

# Developer Setup

## To spin up a local Jenkins instance with shared libraries installed:

```shell
# To bring up Jenkins
docker-compose up --build

# Then visit http://127.0.0.1:8080/

# If you are testing the setBuildStatus* libraries, you'll need to be on JTV VPN (to avoid hitting midway gateway when talking to GHE during testing)
```

## Using the included Dockerfile

The files used for a local Docker run:

* `Dockerfile`
* `docker-compose.yml`
* `docker/usr/share/jenkins/ref/config.xml`
  * Unattended config to set "unauthenticated" and other defaults so we don't have to perform first setup via the UI.
* `docker/usr/share/jenkins/ref/plugins.txt`
  * Version-pinned list of plugins that get installed via install-plugins.sh (https://github.com/jenkinsci/docker/blob/master/README.md#preinstalling-plugins) in our Dockerfile
* `docker/usr/share/jenkins/ref/org.jenkinsci.plugins.workflow.libs.GlobalLibraries.xml`
  * Automatically loads our shared libraries defined here as `flexo-libs` and loads the libraries for all `Jenkinsfile`s implicitly
* `docker/usr/share/jenkins/ref/jobs/*`
  * Job configuration can be stored here.  More details on that below
* `docker/usr/share/jenkins/ref/secrets`
  * When Jenkins spins up, it creates a key to sign secrets with.  We're overriding that key here so that when a secret gets added, its able to be loaded automatically.  Currently the secret `flexo-org-service_pat` is configured so that Jenkins can talk to GHE

## Adding a new library and jenkins test jobs

Let's assume we're working on a new library called `MYTEST`

1. `touch ./docker/Jenkinsfiles/MYTEST_Jenkinsfile`
    * Notes on writing your Jenkinsfile are below
2. `cp -r  ./docker/TEMPLATEJOB ./docker/usr/share/jenkins/ref/jobs/MYTEST_test_pipeline`
3. Open `./docker/usr/share/jenkins/ref/jobs/MYTEST_test_pipeline/config.xml`
    * Find the line that says `<scriptPath>REPLACE_THIS_FILENAME</scriptPath>`
    * Replace `REPLACE_THIS_FILENAME` with your new Jenkinsfile's name - 
        * `<scriptPath>MYTEST_Jenkinsfile</scriptPath>`
4. `docker-compose up --build`
    * Wait for Jenkins to come up, then head to `localhost:8080` and you should see your job!

Edits to `./Jenkinsfiles/MYTEST_Jenkinsfile` will show up in your locally running Jenkins in real time


## Writing your Jenkinsfile

The shared libraries are configured to automatically load in any pipeline, so no need to import the library manually at the top of the `Jenkinsfile`.  Everything should work as expected.


## Unit Testing Jenkins Shared Libraries

I HIGHLY recommend using IntelliJ for your development environment and getting properly set up to work with gradle.

Open the flexo-libs directory in IntelliJ, then follow [these instructions to import a gradle project into intelliJ][intellij_gradle_project]

Tests are under `src/test/groovy/*`

Testing uses the [JenkinsPipelineUnit library][jenkinspipelineunit]

```shell
# To run tests - this assumes you have gradle/groovy/java working in harmony
# Otherwise, just launch the tests from within IntelliJ and your life will be easier
./gradlew clean build test
```

## Further Reading

[Official docs for Jenkins Shared Libraries][jenkins_shared_libs]

[The unit testing library for Jenkins - JenkinsPipelineUnit][jenkinspipelineunit]

[Flexo Jenkins Shared Library Documenation][library_doc]

[intellij_gradle_project]:https://www.jetbrains.com/help/idea/gradle.html#gradle_import_project_start

[jenkins_shared_libs]: https://jenkins.io/doc/book/pipeline/shared-libraries/
[jenkinspipelineunit]: https://github.com/jenkinsci/JenkinsPipelineUnit
[library_doc]: https://wiki.twitch.com/display/PS/Flexo+Jenkins+Shared+Libraries
