package com.twitch.ipdev.flexo.pipelines

import com.twitch.ipdev.flexo.FlexoPipelineBase
import com.twitch.ipdev.flexo.FlexoPipelineStage
import com.twitch.ipdev.flexo.FlexoStage
import com.twitch.ipdev.flexo.pipelines.deploy.DeployECS
import groovy.transform.InheritConstructors
import groovy.lang.Binding

/**
 * Instantiates a .NET Core (C#) pipeline with deploy support for ECS via CodeDeploy.
 * <p>
 * Creates a full Continuous Deployment pipeline that performs the following actions:
 * <ul>
 * <li>Initialize
 * <li>Unit Test
 * <li>Build
 * <li>Publish
 * <li>Staging Deploy
 * <li>Integration Test
 * <li>Approval Workflow
 * <li>Production Deploy
 * </ul>
 * <p>
 * Fields marked as <b>Required</b> must be set in the Closure passed to <em>flexo.pipeline</em> or the pipeline will
 * fail to execute.
 */
@InheritConstructors
class DotnetcoreECS extends FlexoPipelineBase {
  /**
   * <b>Required</b> - IAM Role to assume for publishing our Docker image(s) to ECR
   */
  public String ecrAssumeRoleArn
  /**
   * <b>Required</b> - URL to the ECR Repository to be used for the Publish stage
   */
  public String ecrRepository

  /**
   * AWS Account ID used for Staging Deploy steps
   */
  public String stagingAWSAccount

  /**
   * <b>Required</b> - IAM Role to assume for Staging Deploy of our updated Task Definition to ECS via CodeDeploy
   */
  public String stagingECSAssumeRoleArn

  /**
   * <b>Required</b> - ECS Task Family to use for Staging Deploy
   */
  public String stagingECSTaskFamily

  /**
   * <b>Required</b> - CodeDeploy Application to target for Staging Deploy
   */
  public String stagingCodeDeployApplication

  /**
   * <b>Required</b> - CodeDeploy Deployment Group to target for Staging Deploy
   */
  public String stagingCodeDeployDeploymentGroup

  /**
   * AWS Account ID used for Production Deploy steps
   */
  public String productionAWSAccount

  /**
   * <b>Required</b> - IAM Role to assume for Production Deploy of our updated Task Definition to ECS via CodeDeploy
   */
  public String productionECSAssumeRoleArn

  /**
   * <b>Required</b> - ECS Task Family to use for Production Deploy
   */
  public String productionECSTaskFamily

  /**
   * <b>Required</b> - CodeDeploy Application to target for Production Deploy
   */
  public String productionCodeDeployApplication

  /**
   * <b>Required</b> - CodeDeploy Deployment Group to target for Production Deploy
   */
  public String productionCodeDeployDeploymentGroup

  DotnetcoreECS (Script script, Map args, env) {
    super(script, args, env)
  }

  /**
   * Initialize the pipeline, setting up env vars and defining the contents of each build stage as a Closure that gets
   * executed at runtime.&nbsp;Runs automatically as part of the call to <b>flexo.pipeline</b>.
   */
  public init() {
    super.init()

    // prevent dotnet from trying to write to /.dotnet as the Jenkins user
    env.DOTNET_CLI_HOME = '/tmp/jenkins_dotnet'
    env.DOTNET_CLI_TELEMETRY_OPTOUT = 1
    env.DOCKER_BUILDKIT = 1

    String baseImageName = env.JOB_BASE_NAME.toLowerCase().replaceAll("_","-").replaceAll("[^a-z0-9-]+","")

    // build pipeline
    buildStages['UNIT_TEST'] = new FlexoPipelineStage(script, FlexoStage.UNIT_TEST, reportStatus, {
      String buildImageName = "$baseImageName-test"
      def buildImage = docker.build("$buildImageName:${env.BUILD_ID}", "--target test .").inside {
        sh 'cp /coverage.cobertura.xml coverage.cobertura.xml'
        step([$class: 'CoberturaPublisher', autoUpdateHealth: false, autoUpdateStability: false, coberturaReportFile: '**/coverage.cobertura.xml', failUnhealthy: false, failUnstable: false, maxNumberOfBuilds: 0, onlyStable: false, sourceEncoding: 'ASCII', zoomCoverageChart: false])
      }
    }, env);

    buildStages['BUILD'] = new FlexoPipelineStage(script, FlexoStage.BUILD, reportStatus, {
      def buildImageName = "$baseImageName-build"
      def buildImage = docker.build("$buildImageName:${env.BUILD_ID}", "--target build .")
    }, env);

    buildStages['PUBLISH'] = new FlexoPipelineStage(script, FlexoStage.PUBLISH, reportStatus, {
      def currentDate = new Date();
      env.ecrImage = "$ecrRepository:${currentDate.getTime()}"
      def publishImage = docker.build(env.ecrImage, "--target publish .")

      env.AWS_DEFAULT_REGION = 'us-west-2' // TODO: make region configurable and support multi-region deploys

      // once the image is ready, authenticate with ECR
      // first get temporary credentials via ecrAssumeRoleArn
      def stsJsonOutput = sh(script: "#!/bin/sh -e\naws sts assume-role --role-arn $ecrAssumeRoleArn --role-session-name flexo-ecr-push --duration-seconds 900", returnStdout: true)
      def tempCredentials = readJSON(text: stsJsonOutput)

      // then use those credentials
      env.AWS_ACCESS_KEY_ID = tempCredentials.Credentials.AccessKeyId
      env.AWS_SECRET_ACCESS_KEY = tempCredentials.Credentials.SecretAccessKey
      env.AWS_SESSION_TOKEN = tempCredentials.Credentials.SessionToken

      // debug statement to show that we've assumed role correctly
      sh 'aws sts get-caller-identity'

      // authenticate with ecr via docker
      sh "aws ecr get-login-password | docker login --username AWS --password-stdin $ecrRepository"

      // push the image using docker
      sh "docker push $env.ecrImage"
      sh "rm ${JENKINS_HOME}/.docker/config.json || true"

      env.AWS_ACCESS_KEY_ID = ""
      env.AWS_SECRET_ACCESS_KEY = ""
      env.AWS_SESSION_TOKEN = ""
    }, env);

    // deploy pipeline
    deployStages['STAGING'] = new FlexoPipelineStage(script, FlexoStage.STAGING_DEPLOY, reportStatus,
      DeployECS.deploy(stagingECSAssumeRoleArn, stagingECSTaskFamily, stagingCodeDeployApplication, stagingCodeDeployDeploymentGroup),
      , env);

    deployStages['INTEGRATION'] = new FlexoPipelineStage(script, FlexoStage.INTEGRATION_TEST, reportStatus, {
      echo 'This is where we perform integration tests'
    });

    deployStages['APPROVAL'] = new FlexoPipelineStage(script, FlexoStage.DEPLOY_APPROVAL, reportStatus, {
      timeout(time: 1, unit: 'HOURS') {
        input 'Deploy to Production?'
      }
    }, env);

    deployStages['PRODUCTION'] = new FlexoPipelineStage(script, FlexoStage.PRODUCTION_DEPLOY, reportStatus,
      DeployECS.deploy(productionECSAssumeRoleArn, productionECSTaskFamily, productionCodeDeployApplication, productionCodeDeployDeploymentGroup),
      , env);
  }

  /**
   * Custom validation for the pipeline, checking required field values and string formats for known patterns (AWS ARNs,
   * urls, email addresses, etc.); executed at runtime.&nbsp;Runs automatically as part of the call to <b>flexo.pipeline</b>.
   *
   * @throws Exception
   */
  // TODO: Create custom Exception class FlexoPipelineValidationException
  // TODO: More validation!
  public validate() {
    // Do a simple regex check and for existence of all required variables
    if(!isEcrRepository(ecrRepository)) {throw new Exception("Must set ecrRepository in your Jenkinsfile for a ${this.class.getSimpleName()} Flexo Pipeline")}
    if(!stagingECSTaskFamily) {throw new Exception("Must set stagingECSTaskFamily in your Jenkinsfile for a ${this.class.getSimpleName()} Flexo Pipeline")}
    if(!productionECSTaskFamily) {throw new Exception("Must set productionECSTaskFamily in your Jenkinsfile for a ${this.class.getSimpleName()} Flexo Pipeline")}
    [stagingAWSAccount, productionAWSAccount].each() {
      if(!isAwsAccountId(it)) {throw new Exception("Must set a valid 12 digit AWS account id for ${it} in your Jenkinsfile for a ${this.class.getSimpleName()} Flexo Pipeline")}
    }
    [ecrAssumeRoleArn, stagingECSAssumeRoleArn, productionECSAssumeRoleArn].each() {
    if(!isArn(it)) {throw new Exception("Must set a valid arn for ${it} in your Jenkinsfile for a ${this.class.getSimpleName()} Flexo Pipeline")}
    }

    super.validate()
  }

  private static boolean isArn(String arn) {
    return arn ==~ /arn:aws:[a-z]+::[0-9]{12}:[a-z]+.*/
  }

  private static boolean isAwsAccountId(String id) {
    return id.length() == 12 && id.isInteger()
  }

  private static boolean isEcrRepository(String repo) {
    return repo ==~ /[0-9]{12}\..*amazonaws\.com\/.*/
  }
}
